--[[
	vBagnon\rules.lua
		Helper functions for defining rule functions

		A rule function should take the form of
			function(itemLink)
				return < true | false | nil >
			end

		An itemLink looks like item:%d+:%d+:%d+:%d+, and is the second return of GetItemInfo
		
		vBagnon uses rule functions to define categories of items.
--]]

--[[ ItemLink Data Access ]]--

--returns an item's ID, lower case version of its name, its quality, level, type, subType, and equipLocation
local lastLink
local lastData = {}
function Bagnon_GetData(link)
	if link then
		if link ~= lastLink then
			local name, _, quality, iLevel, level, type, subType, _, loc = GetItemInfo(link)
			if name then
				lastLink = link
				lastData[1] = link
				lastData[2] = name:lower()
				lastData[3] = quality
				lastData[4] = iLevel
				lastData[5] = level
				lastData[6] = type
				lastData[7] = subType
				lastData[8] = loc
			else
				return
			end
		end
		return unpack(lastData)
	end
end

function Bagnon_HyperLinkToLink(hyperLink)
	return hyperLink:match('item:%d+:%d+:%d+:%d+:%d+:%d+:%d+')
end

--[[ Logic ]]--

--returns a function that is equivalent to not f(x)
function Bagnon_Not(f)
	return function(x, y) return not f(x, y) end
end

--returns a function that is equivalent to f1(x) and f2(x) and f3(x) ...
function Bagnon_And(...)
	local args = {...}
	return function(x, y)
		for _, f in ipairs(args) do
			if not f(x, y) then
				return nil
			end
		end
		return true
	end
end

--returns a function g(x) that is equivalent to f1(x) or f2(x) or f3(x) ...
function Bagnon_Or(...)
	local args = {...}
	return function(x, y)
		for _, f in ipairs(args) do
			if f(x, y) then
				return true
			end
		end
		return nil
	end
end

--[[ Tag Logic ]]--

--returns true if the given itemLink is tagged as tag1 and tag2 and tag3 ...
function Bagnon_TaggedAnd(...)
	local args = {...}
	return function(link)
		if link then
			for _, tag in ipairs(args) do
				if not BagnonTag_IsTagged(Bagnon_HyperLinkToLink(link), tag) then
					return nil
				end
			end
			return true
		end
	end
end

--returns true if the given itemLink is taged as either tag1 or tag2 or tag3 ...
function Bagnon_TaggedOr(...)
	local args = {...}
	return function(link)
		if link then
			for _, tag in ipairs(args) do
				if BagnonTag_IsTagged(Bagnon_HyperLinkToLink(link), tag) then
					return true
				end
			end
		end
	end
end

--[[ Rules ]]--

local function ToRuleIndex(s)
	if not s:find('Bagnon') then
		return 'BagnonRule.' .. s
	else
		return s
	end
end

--takes a rule string, which can look like not(and(or(RuleName1, RuleName2), RuleName3))
--and converts it into a function
local stringRules = {}

function Bagnon_StringToRule(ruleStr)
	if ruleStr then
		local rule = stringRules[ruleStr]
		if not rule then		
			if ruleStr:sub(ruleStr:len()) == ')' then
				ruleStr = ruleStr:gsub('not%(', 'Bagnon_Not%(')
				ruleStr = ruleStr:gsub('and%(', 'Bagnon_And%(')
				ruleStr = ruleStr:gsub('or%(', 'Bagnon_Or%(')
				ruleStr = ruleStr:gsub('([%w_]+)', ToRuleIndex)

				rule = assert(loadstring('return ' .. ruleStr))()
			else
				rule = BagnonRule[ruleStr]
			end

			stringRules[ruleStr] = rule
		end
		return rule
	end
end

BagnonRule = {}