--[[
	gChat
	Author: Lars Norberg
	Credits: Baine, Mikma, Nightcracker
	
	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is its designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
]]--

local L = gLib:FetchLocale("gChat", GetLocale());

local defaults = {
	-- non-visible options
	chatlinks 					= true;					-- change Prat chatlinks into clickable links
	chatbottombutton 			= true;					-- always show the bottombutton, but only when not being at the bottom

	-- other options
	chatarrows 					= true;					-- allow the use of the arrowkeys in the inputbox
	chatbox 					= false;				-- move the inputbox to the top
	chatfade 					= true;					-- whether or not the chat will fade out by itself

	cluttervisibility 			= 2; 					-- 0 = hidden, 1 = shown, 2 = hover
	clutterframeposition 		= 0; 					-- 0 = outside, 1 = inside
	clutterscreenposition 		= 0; 					-- 0 = closest to the center, 1 = closest to the edges, 2 = always right, 3 = always left

	editboxmode 				= 2;					-- 0 = hidden, 1 = blizzard, 2 = gUI
		neonchat 				= true; 					-- enable colored border based on chat type
		chatboxhide 			= true;						-- hide the frame surrounding the inputbox

	shortmode 					= 1; 					-- 0 = no abbreviations, 1 = channelnames, loot, xp, rep, 2 = remove brackets as well
		chatshort 				= true;						-- enable abbreviated channel names
		chatbracketshide 		= true; 					-- Show/Hide brackets surrounding channel/player names, requires chatshort = true

	fontmode 					= 1; 					-- 0 = default, 1 = outlined, 2 = outlined + shadow
		chatoutline 			= true;						-- enable chat outline
		chatshadow 				= false;					-- activate the shadow under the text
}

local slash = {
	[1] = {
		name 		= "menu";
		func 		= function() 
			InterfaceOptionsFrame_OpenToCategory(gChat.optionspanel.name); 
		end;
		slashes 	= {"gchat"};
	};
	[2] = {
		name 		= "telltarget";
		func 		= function(msg) 
			gChat:TellTarget(msg, "target");
		end;
		slashes 	= {"tt", "wt"};
	};
	[3] = {
		name 		= "tellfocus";
		func 		= function(msg) 
			gChat:TellTarget(msg, "focus");
		end;
		slashes 	= {"tf", "wf"};
	};
};

gChat_DB = gChat_DB or {};

gChat = gLib:New({
	name = "gChat";
	events = {
		"PLAYER_ENTERING_WORLD";
		"PLAYER_LOGIN";
	};
	userDB = gChat_DB;
	defaultDB = defaults;
	slash = slash;
})

self = gChat;

local _G = _G
local VERSION = self.VERSION;
local ADDON = self.ADDON;
local noop = gLib.System.noop


local function HideBrackets(msg)
	-- remove brackets
	msg = gsub(msg, 	"|Hchannel:(%w+)|h%[(%w+)%]|h", 			"|Hchannel:%1|h%2|h"		);
	msg = gsub(msg, 	"|Hchannel:(%w+)|h%[(%d)%. (%w+)%]|h", 		"|Hchannel:%1|h%2.%3|h"		);
	msg = gsub(msg, 	"|Hplayer:(.+):(.+)|h%[(.+)%]|h%:", 		"|Hplayer:%1:%2|h%3|h:"		);
	return msg
end
local function AbbreviateChannels(msg)
	-- this function is for numbered channels only
	msg = gsub(msg, "|Hchannel:(%w+)|h%[(%d)%. (%w+)%]|h", function(a, b, c)
		if not(L["CHANNELS"]) then L["CHANNELS"] = {} end
		if not(L["CHANNELS"][c]) then
			L["CHANNELS"][c] = gLib.String:Upper(c);
		end   
		return "|Hchannel:"..a.."|h"..b.."."..L["CHANNELS"][c].."|h";
	end)

	-- abbreviate the achievement spam
	-- did not find a global string for this :(
	msg = gsub(msg, L["FULL_ACHIEVEMENT_GAINED"], 		L["SHORT_ACHIEVEMENT_GAINED"]);

	return msg
end
local function SetAbbreviations(state)
	-- if our local copies of the original defaults aren't yet made, make them!
	if not(L["DEFAULT_ABBREVIATIONS"]) then
		L["DEFAULT_ABBREVIATIONS"] = {}
		for i,v in pairs(L["ABBREVIATIONS"]) do
			L["DEFAULT_ABBREVIATIONS"][i] = _G[i];
		end
	end
	if state then
		-- replace the dreaded globals with our versions
		for i,v in pairs(L["ABBREVIATIONS"]) do
			_G[i] = L["ABBREVIATIONS"][i];
		end
	else
		-- set the globals back to their defaults
		for i,v in pairs(L["ABBREVIATIONS"]) do
			_G[i] = L["DEFAULT_ABBREVIATIONS"][i];
		end
	end
end
local function RoleIcons(msg)
	-- credits to nightcracker for this part
	local player = gsub(msg, "|Hplayer:([^:]+):.+|h.+", "%1")
	if player and UnitInParty(player) then
		local tank, heal, dps = UnitGroupRolesAssigned(player)		
		msg = ((tank and gMedia.lib.iconstring["role-tank"]) or (heal and gMedia.lib.iconstring["role-heal"]) or (dps and gMedia.lib.iconstring["role-dps"]) or "")..msg
	end
	return msg;
end
local function Clink(msg)
	-- Makes Prat links clickable without having Prat installed
	-- Original patterns retrieved from Prat 3.0 RC8 by Sylvanaar, Dec 1st 2008. (Slightly modified to fit gChat)
	msg = gsub (msg, "{CLINK:achievement:(%x+):(%-?%d-:%x-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-):([^}]-)}", "|c%1|Hachievement:%2|h[%3]|h|r");
	msg = gsub (msg, "{CLINK:trade:(%x+):(%-?%d-:%-?%d-:%-?%d-:%x-:[^}:]+):([^}]-)}","|c%1|Htrade:%2|h[%3]|h|r");
	msg = gsub (msg, "{CLINK:quest:(%x+):(%-?%d-):(%-?%d-):([^}]-)}", "|c%1|Hquest:%2:%3|h%4|h|r");
	msg = gsub (msg, "{CLINK:item:(%x+):(%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-:%-?%d-):([^}]-)}", "|c%1|Hitem:%2|h%[%3%]|h|r");
	msg = gsub (msg, "{CLINK:enchant:(%x+):(%-?%d-):([^}]-)}", "|c%1|Henchant:%2|h%[%3%]|h|r");
	msg = gsub (msg, "{CLINK:spell:(%x+):(%-?%d-):([^}]-)}", "|c%1|Hspell:%2|h%[%3%]|h|r");
	return msg;
end

local function SetSticky()
	-- make pretty much everything sticky, so you can press ENTER to directly speak in Trade, Officer, etc etc
	local stickies = {"SAY", "PARTY", "RAID", "RAID_WARNING", "GUILD", "OFFICER", "BATTLEGROUND", "CHANNEL", "CHANNEL1", "CHANNEL2", "CHANNEL3", "CHANNEL4", "CHANNEL5", "CHANNEL6", "CHANNEL7", "CHANNEL8", "CHANNEL9", "CHANNEL10"}
	for i,v in pairs(stickies) do ChatTypeInfo[v].sticky = 1; end
	
end

local function Parse(args)
	local parse = gLib.Object:ArgumentCheck( args, {
		msg 				= "";
		pratlinks 			= true;
		justifyH 			= "LEFT";
		hidebrackets 		= false;
		roleicons 			= true;
		abbreviate 			= false;
	})

	-- debugging, break all chat links
	-- parse.msg = gsub(parse.msg, "|", "||")

	-- the order of the parses is important, don't change it, or the search patterns and replaces will break!
	if parse.roleicons then 		parse.msg = RoleIcons(parse.msg); end
	if parse.abbreviate then 		parse.msg = AbbreviateChannels(parse.msg); end
	if parse.hidebrackets then 		parse.msg = HideBrackets(parse.msg); end
	if parse.pratlinks then 		parse.msg = Clink(parse.msg); end
	
	return parse.msg;
end
local function AddMessage(msg,a1,a2,a3,a4,a5,a6,a7,a8,a9,...)
	if (msg == nil) then return; end
	msg = Parse({ 
		msg 				= msg; 
		abbreviate 			= gChat_DB.chatshort; 
		hidebrackets 		= gChat_DB.chatbracketshide; 
		justifyH 			= self:GetJustifyH(); 
		pratlinks 			= gChat_DB.chatlinks; 
	});
	self:BlizzAddMessage(msg,a1,a2,a3,a4,a5,a6,a7,a8,a9);
end

local function ToggleNeon(state)
	for cf = 1,NUM_CHAT_WINDOWS,1 do
		local editbox = gChat["ChatFrame"..cf.."EditBox"];
		if state then
			editbox:Show();
		else
			editbox:Hide();
		end
	end
end
local function NeonChat(frame)
	local gbox = gChat[frame:GetName()];
	local function colorize(r,g,b)
 		gbox:SetBackdropBorderColor(r, g, b, 1);
		gbox:SetBackdropColor(r*0.15, g*0.15, b*0.15, 1);
		gbox.shadow:SetBackdropBorderColor(r, g, b, 0.33);
	end
	if gChat_DB.neonchat then
		local type = frame:GetAttribute("chatType")
		if ( type == "CHANNEL" ) then
			local id = GetChannelName(frame:GetAttribute("channelTarget"))
			if id == 0 then	
				colorize(0.5,0.5,0.5)
			else 
				colorize(ChatTypeInfo[type..id].r, ChatTypeInfo[type..id].g, ChatTypeInfo[type..id].b)
			end
		else 
			colorize(ChatTypeInfo[type].r, ChatTypeInfo[type].g, ChatTypeInfo[type].b)
		end
	end
end

local function FadeInChatFrame(chatFrame)
	local frameName = chatFrame:GetName();
	chatFrame.hasBeenFaded = true;
	if ( chatFrame == FCFDock_GetSelectedWindow(GENERAL_CHAT_DOCK) ) then
		for _, frame in pairs(FCFDock_GetChatFrames(GENERAL_CHAT_DOCK)) do
			if ( frame ~= chatFrame ) then
				FCF_FadeInChatFrame(frame);
			end
		end
		if ( GENERAL_CHAT_DOCK.overflowButton:IsShown() ) then
			UIFrameFadeIn(GENERAL_CHAT_DOCK.overflowButton, 0.5, GENERAL_CHAT_DOCK.overflowButton:GetAlpha(), 1);
		end
	end
	
	local chatTab = _G[frameName.."Tab"];
	UIFrameFadeIn(chatTab, 0.5, chatTab:GetAlpha(), 1);
	
	--Fade in the button frame
	if ( not chatFrame.isDocked ) then
		UIFrameFadeIn(chatFrame.buttonFrame, 0.5, chatFrame.buttonFrame:GetAlpha(), 1);
	end
end
local function FadeOutChatFrame(chatFrame)
	local frameName = chatFrame:GetName();
	chatFrame.hasBeenFaded = nil;
	for index, value in pairs(CHAT_FRAME_TEXTURES) do
		-- Fade out chat frame
		local object = _G[frameName..value];
		if ( object:IsShown() ) then
			UIFrameFadeOut(object, 0.5, max(object:GetAlpha(), chatFrame.oldAlpha), 0);
		end
	end
	if ( chatFrame == FCFDock_GetSelectedWindow(GENERAL_CHAT_DOCK) ) then
		for _, frame in pairs(FCFDock_GetChatFrames(GENERAL_CHAT_DOCK)) do
			if ( frame ~= chatFrame ) then
				FCF_FadeOutChatFrame(frame);
			end
		end
		if ( GENERAL_CHAT_DOCK.overflowButton:IsShown() ) then
			UIFrameFadeOut(GENERAL_CHAT_DOCK.overflowButton, 0.5, GENERAL_CHAT_DOCK.overflowButton:GetAlpha(), (gChat_DB.cluttervisibility == 1) and CHAT_FRAME_TAB_SELECTED_NOMOUSE_ALPHA or 0);
		end
	end
	
	local chatTab = _G[frameName.."Tab"];
	UIFrameFadeOut(chatTab, 0.5, chatTab:GetAlpha(), (gChat_DB.cluttervisibility == 1) and chatTab.noMouseAlpha or 0);
	
	--Fade out the ButtonFrame
	if ( not chatFrame.isDocked ) then
		UIFrameFadeOut(chatFrame.buttonFrame, 1, chatFrame.buttonFrame:GetAlpha(), (gChat_DB.cluttervisibility == 1) and CHAT_FRAME_BUTTON_FRAME_MIN_ALPHA or 0);
	end
end
local function SetClutterDisplayMode()
	local cmode = gChat_DB.cluttervisibility
	local func = {
		-- always hidden
		[0] = function(frame) 
			UIFrameFadeOut(frame, 0.5, frame:GetAlpha(), 0); 
		end;

		-- always shown
		[1] = function(frame) 
			frame:Show();
			UIFrameFadeIn(frame, 0.5, frame:GetAlpha(), 1); 
		end;

		-- on hover
		[2] = function(frame) 
			UIFrameFadeOut(frame, 0.5, frame:GetAlpha(), 0); 
		end;
	}
	
	for i = 1, NUM_CHAT_WINDOWS, 1 do
		local elements = { _G["ChatFrame"..i.."ButtonFrameUpButton"], _G["ChatFrame"..i.."ButtonFrameDownButton"], _G["ChatFrame"..i.."ButtonFrameBottomButton"] }
		for j,w in pairs(elements) do
			func[cmode](w);
		end
		if (_G["ChatFrame"..i] ~= DEFAULT_CHAT_FRAME) and not(_G["ChatFrame"..i].isDocked) then
			func[cmode](_G["ChatFrame"..i.."ButtonFrameMinimizeButton"]);
		else
			func[cmode](ChatFrameMenuButton);
			func[cmode](FriendsMicroButton);
		end
	end
end
local function SetButtonSide(frame, side, forceUpdate)

	if (( frame.buttonSide == side  ) and (oldmode == cmode) and not(forceUpdate) ) then
		return;
	end

	local cmode = gChat_DB.clutterframeposition
	local oldmode = frame.cluttermode
	local p = 4;
	local topY = 0;

	if ( IsCombatLog(frame) ) then
		topY = topY + CombatLogQuickButtonFrame_Custom:GetHeight();
	end
	
	frame.buttonFrame:ClearAllPoints();

	if ( side == "left" ) then
		if (cmode == 0) then
			-- outside the frame!
			frame.buttonFrame:SetPoint("TOPRIGHT", frame, "TOPLEFT", -p, topY);
			frame.buttonFrame:SetPoint("BOTTOMRIGHT", frame, "BOTTOMLEFT", -p, 0);
		else
			-- inside the frame?
			frame.buttonFrame:SetPoint("TOPLEFT", frame, "TOPLEFT", w + p, topY);
			frame.buttonFrame:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", w + p, 0);
		end
	elseif ( side == "right" ) then
		if (cmode == 0) then
			-- outside the frame!
			frame.buttonFrame:SetPoint("TOPLEFT", frame, "TOPRIGHT", p, topY);
			frame.buttonFrame:SetPoint("BOTTOMLEFT", frame, "BOTTOMRIGHT", p, 0);
		else
			-- inside the frame?
			frame.buttonFrame:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -p, topY);
			frame.buttonFrame:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -p, 0);
		end
	end

	if ( frame == DEFAULT_CHAT_FRAME ) then
		ChatFrameMenu_UpdateAnchorPoint();
	end
	
	frame.cluttermode = cmode;
	frame.buttonSide = side;
end
local function UpdateButtonSide(chatframe)
	if not(chatframe) then return end
	
	local leftDist = chatframe:GetLeft();
	local rightDist = GetScreenWidth() - chatframe:GetRight();
	local changed = nil;
	local cmode = gChat_DB.clutterscreenposition;
	
	-- at the center
	if (cmode == 0) then
		if (leftDist <= rightDist) then 
			if (chatframe.buttonSide ~= "right") then
				FCF_SetButtonSide(chatframe, "right");
				changed = 1;
			end
		elseif (chatframe.buttonSide ~= "left") then
			FCF_SetButtonSide(chatframe, "left");
			changed = 1;
		end

		-- at the edges
	elseif (cmode == 1) then
		if (leftDist >= rightDist) then 
			if (chatframe.buttonSide ~= "right") then
				FCF_SetButtonSide(chatframe, "right");
				changed = 1;
			end
		elseif (chatframe.buttonSide ~= "left") then
			FCF_SetButtonSide(chatframe, "left");
			changed = 1;
		end

		-- always right
	elseif (cmode == 2) then
		if (chatframe.buttonSide ~= "right") then
			FCF_SetButtonSide(chatframe, "right");
			changed = 1;
		end
	
	-- always left
	elseif (cmode == 3) then
		if (chatframe.buttonSide ~= "left") then 
			FCF_SetButtonSide(chatframe, "left");
			changed = 1;
		end

	end
	return changed;
end
local function UpdateAllButtonSides()
	for cf = 1,NUM_CHAT_WINDOWS,1 do
		local frame = _G["ChatFrame"..cf];
		FCF_UpdateButtonSide(frame);
		FCF_SetButtonSide(frame, FCF_GetButtonSide(frame));
	end
end

local function SetEditBoxPosition(state)
	local height, padding = 22, UIPADDING or 8
	for cf = 1,NUM_CHAT_WINDOWS,1 do
		local frame = _G["ChatFrame"..cf.."EditBox"];
		local cframe = _G["ChatFrame"..cf];
	
		if state then 
			frame:ClearAllPoints()
			frame:SetPoint("BOTTOMLEFT", cframe, "TOPLEFT", -4, padding + 18);
			frame:SetPoint("BOTTOMRIGHT", cframe, "TOPRIGHT", 4, padding + 18 );
			frame:SetPoint("TOPLEFT", cframe, "TOPLEFT", -4, height + padding + 18);
			frame:SetPoint("TOPRIGHT", cframe, "TOPRIGHT", 4, height + padding + 18);
		else
			frame:ClearAllPoints()
			frame:SetPoint("BOTTOMLEFT", cframe, "BOTTOMLEFT", -4, -(height + padding) + 18);
			frame:SetPoint("BOTTOMRIGHT", cframe, "BOTTOMRIGHT", 4, -(height + padding) + 18);
			frame:SetPoint("TOPLEFT", cframe, "BOTTOMLEFT", -4, -padding + 18);
			frame:SetPoint("TOPRIGHT", cframe, "BOTTOMRIGHT", 4, -padding + 18);
		end
	end
end
local function EditBoxAlpha(state)
	for cf = 1,NUM_CHAT_WINDOWS,1 do
		local frame = _G["ChatFrame"..cf.."EditBox"];
		local x=({frame:GetRegions()}); 
		if state then
			-- editbox border
			if x[6] then x[6]:Hide(); end
			if x[7] then x[7]:Hide(); end
			if x[8] then x[8]:Hide(); end
			if x[9] then 
				x[9]:Hide();
				x[9].OldShow = x[9].Show;
				x[9].Show = noop; 
			end
			if x[10] then 
				x[10]:Hide();
				x[10].OldShow = x[10].Show;
				x[10].Show = noop; 
			end
			if x[11] then 
				x[11]:Hide();
				x[11].OldShow = x[11].Show;
				x[11].Show = noop; 
			end
		else
			if x[6] then x[6]:Show(); end
			if x[7] then x[7]:Show(); end
			if x[8] then x[8]:Show(); end
			if x[9] then 
				if x[9].OldShow then
					x[9].Show = x[9].OldShow; 
				end
				x[9]:Show(); 
			end
			if x[10] then 
				if x[10].OldShow then
					x[10].Show = x[10].OldShow; 
				end
				x[10]:Show(); 
			end
			if x[11] then 
				if x[11].OldShow then
					x[11].Show = x[11].OldShow; 
				end
				x[11]:Show(); 
			end
		end
	end
end
local function SetArrows(state)
	for cf = 1,NUM_CHAT_WINDOWS,1 do
		_G["ChatFrame"..cf.."EditBox"]:SetAltArrowKeyMode(not state);
	end
end

local function SetFade(state)
	for i = 1,NUM_CHAT_WINDOWS,1 do 
		_G["ChatFrame"..i]:SetFading(state);
	end
end
local function SetShadow(state)
	for i = 1,NUM_CHAT_WINDOWS,1 do 
		local f = _G["ChatFrame"..i]; 
		if state then
			fontset,fontheight,fontflags=f:GetFont();
			f:SetShadowColor(0,0,0,0.75);  
			f:SetShadowOffset(2,-2); 
		else
			fontset,fontheight,fontflags=f:GetFont();
			f:SetShadowColor(0,0,0,0.75);  
			f:SetShadowOffset(1.25,-1.25); 
		end
	end
end
local function SetOutline(state)
	for i = 1,NUM_CHAT_WINDOWS,1 do 
		local f = _G["ChatFrame"..i]; 
		if state then
			fontset,fontheight,fontflags=f:GetFont();
			f:SetFont(fontset,fontheight , "OUTLINE");
		else
			fontset,fontheight,fontflags=f:GetFont();
			f:SetFont(fontset,fontheight , "");
		end
	end
end

function gChat:ChatHover(frame)
	local vmode = gChat_DB.cluttervisibility

	local side = frame.buttonSide;
	local cmode = frame.cluttermode;
	
	local left = (cmode == 0) and (((side == "left") and -37) or 37) or 0;
	local right = (cmode == 0) and (((side == "right") and 37) or -37) or 0;

	if (vmode == 2) then
		if MouseIsOver(frame, 37, 0, -37, 37) then
			_G[frame:GetName().."ButtonFrameUpButton"]:Show();
			_G[frame:GetName().."ButtonFrameDownButton"]:Show();
			UIFrameFadeIn(_G[frame:GetName().."ButtonFrameUpButton"], 0.5, _G[frame:GetName().."ButtonFrameUpButton"]:GetAlpha(), 1);
			UIFrameFadeIn(_G[frame:GetName().."ButtonFrameDownButton"], 0.5, _G[frame:GetName().."ButtonFrameDownButton"]:GetAlpha(), 1);
			if (frame == DEFAULT_CHAT_FRAME) or (frame.isDocked) then 
				ChatFrameMenuButton:Show();
				FriendsMicroButton:Show();
				UIFrameFadeIn(ChatFrameMenuButton, 0.5, ChatFrameMenuButton:GetAlpha(), 1);
				UIFrameFadeIn(FriendsMicroButton, 0.5, ChatFrameMenuButton:GetAlpha(), 1);
			else
				_G[frame:GetName().."ButtonFrameMinimizeButton"]:Show();
				UIFrameFadeIn(_G[frame:GetName().."ButtonFrameMinimizeButton"], 0.5, _G[frame:GetName().."ButtonFrameMinimizeButton"]:GetAlpha(), 1);
			end
		else
			UIFrameFadeOut(_G[frame:GetName().."ButtonFrameUpButton"], 0.5, _G[frame:GetName().."ButtonFrameUpButton"]:GetAlpha(), 0);
			UIFrameFadeOut(_G[frame:GetName().."ButtonFrameDownButton"], 0.5, _G[frame:GetName().."ButtonFrameDownButton"]:GetAlpha(), 0);
			UIFrameFadeOut(_G[frame:GetName().."ButtonFrameMinimizeButton"], 0.5, _G[frame:GetName().."ButtonFrameMinimizeButton"]:GetAlpha(), 0);
			UIFrameFadeOut(ChatFrameMenuButton, 0.5, ChatFrameMenuButton:GetAlpha(), 0);
			UIFrameFadeOut(FriendsMicroButton, 0.5, FriendsMicroButton:GetAlpha(), 0);
		end
	end
	if frame:AtBottom() then
		UIFrameFadeOut(_G[frame:GetName().."ButtonFrameBottomButton"], 0.5, _G[frame:GetName().."ButtonFrameBottomButton"]:GetAlpha(), 0);
	else
		_G[frame:GetName().."ButtonFrameBottomButton"]:Show();
		UIFrameFadeIn(_G[frame:GetName().."ButtonFrameBottomButton"], 0.5, _G[frame:GetName().."ButtonFrameBottomButton"]:GetAlpha(), 1);
	end
end
function gChat:ThreeThreeFive()
	local frame
	for cf = 1,NUM_CHAT_WINDOWS,1 do
		frame = _G["ChatFrame"..cf];
		frame:SetClampedToScreen(false);
	end
end

function gChat:UpdateAll()
	-- make extra sure everything has a value
	gChat_DB = gLib.Object:ArgumentCheck( gChat_DB, self.defaults);
	
	self:ThreeThreeFive();

	if gChat_DB.editboxmode == 0 then
		gChat_DB.neonchat = false; 
		gChat_DB.chatboxhide = true;
		
	elseif gChat_DB.editboxmode == 1 then
		gChat_DB.neonchat = false; 
		gChat_DB.chatboxhide = false;
		
	elseif gChat_DB.editboxmode == 2 then
		gChat_DB.neonchat = false; 
		gChat_DB.chatboxhide = true;
	end
	
	if gChat_DB.shortmode == 0 then
		gChat_DB.chatshort = false;
		gChat_DB.chatbracketshide = false;
		
	elseif gChat_DB.shortmode == 1 then 
		gChat_DB.chatshort = true;
		gChat_DB.chatbracketshide = false;
		
	elseif gChat_DB.shortmode == 2 then 
		gChat_DB.chatshort = true;
		gChat_DB.chatbracketshide = true;
	end
	
	if gChat_DB.fontmode == 0 then
		gChat_DB.chatoutline = false;
		gChat_DB.chatshadow = false;
		
	elseif gChat_DB.fontmode == 1 then 
		gChat_DB.chatoutline = true;
		gChat_DB.chatshadow = false;
		
	elseif gChat_DB.fontmode == 2 then 
		gChat_DB.chatoutline = true;
		gChat_DB.chatshadow = true;
	end
	
	SetEditBoxPosition(gChat_DB.chatbox);
	SetArrows(gChat_DB.chatarrows);
	SetFade(gChat_DB.chatfade);
	SetShadow(gChat_DB.chatshadow);
	SetOutline(gChat_DB.chatoutline);
	ToggleNeon(gChat_DB.neonchat);
	EditBoxAlpha(gChat_DB.chatboxhide);
	SetAbbreviations(gChat_DB.chatshort);
	SetSticky();

	UpdateAllButtonSides();
	SetClutterDisplayMode();

end

function gChat:TellTarget(txt, who)
  if UnitExists(who) and UnitIsPlayer(who) and UnitIsFriend("player", who) then
    local name = gChat:GetCharName(who)
	SendChatMessage(txt, "WHISPER", nil, name)    
    ChatEdit_SetLastToldTarget(name)	
  end
end
function gChat:TellTargetKeyBind(who)
	local editBox = DEFAULT_CHAT_FRAME.editBox
	local text = editBox:GetText()

	editBox:Show()
	editBox:SetAttribute("chatType", "WHISPER")
	if UnitExists(who) and UnitIsPlayer(who) and UnitIsFriend("player", who) then
		local name = gChat:GetCharName(who)	
		editBox:SetAttribute("tellTarget", name)
		ChatEdit_UpdateHeader(editBox)
	else
		editBox:SetAttribute("tellTarget", who)
		if( who == "target") then
			editBox:SetText("/tt ")	
		elseif( who == "focus") then
			editBox:SetText("/tf ")
		end
		ChatEdit_UpdateHeader(editBox)
	end
end
function gChat:OnSpacePressed()
	local editBox = DEFAULT_CHAT_FRAME.editBox;
	local text = editBox:GetText()
	local who = "target"
	
	if (string.len(text) ~= 4) then return end
	if (strsub(text, 1, 1) ~= "/") then return end
	
	local slashcmd = strsub(text, 2, 3)
	if slashcmd == "wf" or slashcmd == "tf" then	
		who = "focus"; 
	elseif slashcmd ~= "wt" and slashcmd ~= "tt" then
		return
	end
	
	editBox:SetAttribute("chatType", "WHISPER")
	if UnitExists(who) and UnitIsPlayer(who) and UnitIsFriend("player", who) then
		local name = gChat:GetCharName(who)	
		editBox:SetText("")
		editBox:SetAttribute("tellTarget", name)
		ChatEdit_UpdateHeader(editBox)
	else
		editBox:SetAttribute("tellTarget", who)
		ChatEdit_UpdateHeader(editBox)
	end
end
function gChat:GetCharName(who)
	local targetName, targetServerName = UnitName(who)
	
	if not targetServerName or (#targetServerName < 1) then
		return targetName
		
	elseif targetServerName ~= gChat.REALM then
	    local crossServerName = targetName.."-"..targetServerName
		return crossServerName
	end
end

function gChat:PLAYER_LOGIN()
	self:UnregisterEvent("PLAYER_LOGIN")

	BINDING_HEADER_GCHAT = self.NAME;
	BINDING_NAME_GCHAT_TELLTARGET = L["KEYBIND_WHISPER_TARGET"];
	BINDING_NAME_GCHAT_TELLFOCUS = L["KEYBIND_WHISPER_FOCUS"];

	FCF_UpdateButtonSide 	= function(frame) UpdateButtonSide(frame) end
	FCF_SetButtonSide 		= function(frame, side) SetButtonSide(frame, side); end 
	FCF_FadeInChatFrame 	= function(frame) FadeInChatFrame(frame); end
	FCF_FadeOutChatFrame 	= function(frame) FadeOutChatFrame(frame); end

	hooksecurefunc("ChatEdit_UpdateHeader", function(...) NeonChat(...); end );
	hooksecurefunc("ChatEdit_OnSpacePressed", gChat.OnSpacePressed );

	for i = 1, NUM_CHAT_WINDOWS, 1 do 
		local frame = _G["ChatFrame"..i]; 
		local name = "ChatFrame"..i.."EditBox";
		local editbox = _G[name];

		self[name] = gMedia:panel({ parent = editbox; framestrata = editbox:GetFrameStrata(); });
		self[name]:SetAllPoints(editbox);
		self[name]:SetAlpha(1);

		frame.BlizzAddMessage = frame.AddMessage;
		frame.AddMessage = self.Chat_AddMessage;
		
		frame:HookScript("OnUpdate", function(self) gChat:ChatHover(self); end );
	end
	

 	self.optionspanel = gLib.Menu:CreateOptionsPanel({
		title = L["TITLE"];
		version = VERSION;
		description = "";
		panel = ((IsAddOnLoaded("gUI") and gUI.optionspanel) or nil);
		option = {
			[40] = {
				name = L["GUI_CLUTTER_SCRPOS"];
				type = "dropdown";
				value = gChat_DB.clutterscreenposition;
				list = {
					{
						level 		= 1;
						text 		= function() 
							return ((gChat_DB.clutterscreenposition == 0) and L["GUI_CLUTTER_SCRPOS_OPTION_CENTER"]) or ((gChat_DB.clutterscreenposition == 1) and L["GUI_CLUTTER_SCRPOS_OPTION_EDGE"]) or ((gChat_DB.clutterscreenposition == 2) and L["GUI_CLUTTER_SCRPOS_OPTION_RIGHT"]) or ((gChat_DB.clutterscreenposition == 3) and L["GUI_CLUTTER_SCRPOS_OPTION_LEFT"]) or "";
						end;
						menulist = {
							{
								text 			= L["GUI_CLUTTER_SCRPOS_CENTER"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.clutterscreenposition = 0;
									UpdateAllButtonSides();
								end;
							};
							{
								text 			= L["GUI_CLUTTER_SCRPOS_EDGE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.clutterscreenposition = 1;
									UpdateAllButtonSides();
								end;
							};
							{
								text 			= L["GUI_CLUTTER_SCRPOS_RIGHT"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.clutterscreenposition = 2;
									UpdateAllButtonSides();
								end;
							};
							{
								text 			= L["GUI_CLUTTER_SCRPOS_LEFT"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.clutterscreenposition = 3;
									UpdateAllButtonSides();
								end;
							};
						};
					};
				};
			};
			[45] = {
				name = L["GUI_CLUTTER_POSITION"];
				type = "dropdown";
				value = gChat_DB.clutterframeposition;
				list = {
					{
						level 		= 1;
						text 		= function() 
							return ((gChat_DB.clutterframeposition == 0) and L["GUI_CLUTTER_POSITION_OPTION_OUTSIDE"]) or ((gChat_DB.clutterframeposition == 1) and L["GUI_CLUTTER_POSITION_OPTION_INSIDE"]) or "";
						end;
						menulist = {
							{
								text 			= L["GUI_CLUTTER_POSITION_OUTSIDE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.clutterframeposition = 0;
									UpdateAllButtonSides();
								end;
							};
							{
								text 			= L["GUI_CLUTTER_POSITION_INSIDE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.clutterframeposition = 1;
									UpdateAllButtonSides();
								end;
							};
						};
					};
				};
			};
			[35] = {
				name = L["GUI_CLUTTER_VISIBILITY"];
				type = "dropdown";
				value = gChat_DB.cluttervisibility;
				list = {
					{
						level 		= 1;
						text 		= function() 
							return ((gChat_DB.cluttervisibility == 0) and L["GUI_CLUTTER_VISIBILITY_OPTION_HIDDEN"]) or ((gChat_DB.cluttervisibility == 1) and L["GUI_CLUTTER_VISIBILITY_OPTION_VISIBLE"]) or ((gChat_DB.cluttervisibility == 2) and L["GUI_CLUTTER_VISIBILITY_OPTION_HOVER"]) or "";
						end;
						menulist = {
							{
								text 			= L["GUI_CLUTTER_VISIBILITY_HIDDEN"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.cluttervisibility = 0;
									SetClutterDisplayMode();
								end;
							};
							{
								text 			= L["GUI_CLUTTER_VISIBILITY_VISIBLE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.cluttervisibility = 1;
									SetClutterDisplayMode();
								end;
							};
							{
								text 			= L["GUI_CLUTTER_VISIBILITY_HOVER"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.cluttervisibility = 2;
									SetClutterDisplayMode();
								end;
							};
						};
					};
				};
			};
			[30] = {
				name = L["GUI_EDITBOX_MODE"];
				type = "dropdown";
				value = gChat_DB.editboxmode;
				list = {
					{
						level = 1;
						text = function() 
							return ((gChat_DB.editboxmode == 0) and L["GUI_EDITBOX_MODE_OPTION_HIDDEN"]) or ((gChat_DB.editboxmode == 1) and L["GUI_EDITBOX_MODE_OPTION_BLIZZARD"]) or ((gChat_DB.editboxmode == 2) and L["GUI_EDITBOX_MODE_OPTION_NEON"]) or "";
						end;
						menulist = {
							{
								text = L["GUI_EDITBOX_MODE_HIDDEN"];
								notCheckable = 1;
								func = function() 
									gChat_DB.editboxmode = 0;
									gChat_DB.neonchat = false; 
									gChat_DB.chatboxhide = true;
									EditBoxAlpha(gChat_DB.chatboxhide);
									ToggleNeon(gChat_DB.neonchat);
								end;
							};
							{
								text = L["GUI_EDITBOX_MODE_BLIZZARD"];
								notCheckable = 1;
								func = function() 
									gChat_DB.editboxmode = 1;
									gChat_DB.neonchat = false; 
									gChat_DB.chatboxhide = false;
									EditBoxAlpha(gChat_DB.chatboxhide);
									ToggleNeon(gChat_DB.neonchat);
								end;
							};
							{
								text = L["GUI_EDITBOX_MODE_NEON"];
								notCheckable = 1;
								func = function() 
									gChat_DB.editboxmode = 2;
									gChat_DB.neonchat = true; 
									gChat_DB.chatboxhide = true;
									EditBoxAlpha(gChat_DB.chatboxhide);
									ToggleNeon(gChat_DB.neonchat);
								end;
							};
						};
					};
				};
			};
			[10] = {
				name = L["GUI_SHORTS"];
				type = "dropdown";
				value = gChat_DB.shortmode;
				list = {
					{
						level 		= 1;
						text 		= function() 
							return ((gChat_DB.shortmode == 0) and L["GUI_SHORTS_OPTION_NONE"]) or ((gChat_DB.shortmode == 1) and L["GUI_SHORTS_OPTION_CHAN"]) or ((gChat_DB.shortmode == 2) and L["GUI_SHORTS_OPTION_CHANPLAY"]) or "";
						end;
						menulist = {
							{
								text 			= L["GUI_SHORTS_NONE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.shortmode = 0;
									gChat_DB.chatshort = false; 
									gChat_DB.chatbracketshide = false;
									SetAbbreviations(gChat_DB.chatshort); 
								end;
							};
							{
								text 			= L["GUI_SHORTS_CHAN"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.shortmode = 1;
									gChat_DB.chatshort = true; 
									gChat_DB.chatbracketshide = false;
									SetAbbreviations(gChat_DB.chatshort); 
								end;
							};
							{
								text 			= L["GUI_SHORTS_CHANPLAY"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.shortmode = 2;
									gChat_DB.chatshort = true; 
									gChat_DB.chatbracketshide = true;
									SetAbbreviations(gChat_DB.chatshort); 
								end;
							};
						};
					};
				};
			};
			[60] = {
				name = L["GUI_FONT"];
				type = "dropdown";
				value = gChat_DB.fontmode;
				list = {
					{
						level 		= 1;
						text 		= function() 
							return ((gChat_DB.fontmode == 0) and L["GUI_FONT_OPTION_NONE"]) or ((gChat_DB.fontmode == 1) and L["GUI_FONT_OPTION_OUTLINE"]) or ((gChat_DB.fontmode == 2) and L["GUI_FONT_OPTION_SHADOW"]) or "";
						end;
						menulist = {
							{
								text 			= L["GUI_FONT_NONE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.fontmode = 0;
									gChat_DB.chatoutline = false; 
									gChat_DB.chatshadow = false;
									SetShadow(gChat_DB.chatshadow);
									SetOutline(gChat_DB.chatoutline);
								end;
							};
							{
								text 			= L["GUI_FONT_OUTLINE"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.fontmode = 1;
									gChat_DB.chatoutline = true; 
									gChat_DB.chatshadow = false;
									SetShadow(gChat_DB.chatshadow);
									SetOutline(gChat_DB.chatoutline);
								end;
							};
							{
								text 			= L["GUI_FONT_SHADOW"];
								notCheckable 	= 1;
								func = function() 
									gChat_DB.fontmode = 2;
									gChat_DB.chatoutline = true; 
									gChat_DB.chatshadow = true;
									SetShadow(gChat_DB.chatshadow);
									SetOutline(gChat_DB.chatoutline);
								end;
							};
						};
					};
				};
			};
			[100] = {
				name = L["GUI_CHAT_MOVEEDITBOX"];
				type = "button";
				value = gChat_DB.chatbox;
				panelfunction = function() gChat_DB.chatbox = not gChat_DB.chatbox; SetEditBoxPosition(gChat_DB.chatbox); end;
			};
			[120] = {
				name = L["GUI_CHAT_ENABLEARROWS"];
				type = "button";
				value = gChat_DB.chatarrows;
				panelfunction = function() gChat_DB.chatarrows = not gChat_DB.chatarrows; SetArrows(gChat_DB.chatarrows); end;
			};
			[130] = {
				name = L["GUI_CHAT_FADE"];
				type = "button";
				value = gChat_DB.chatfade;
				panelfunction = function() gChat_DB.chatfade = not gChat_DB.chatfade; SetFade(gChat_DB.chatfade); end;
			};
		};
	}, gChat_DB, self.defaults);

end
function gChat:PLAYER_ENTERING_WORLD()
	self:UpdateAll();
end

