--[[
	gLib
	Author: Lars Norberg
	
	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is its designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
]]--

-- this is the current version number of gLib. DO NOT MODIFY!!
local version = 2.4

-- embedding ftw
local parent = debugstack():match[[\AddOns\(.-)\]]
if (gLib) and (gLib.VERSION) and (tonumber(gLib.VERSION) >= tonumber(version)) then return end

gLib 					= {};
gLib.ADDON 				= parent or "gLib"; 
gLib.VERSION 			= version or GetAddOnMetadata(ADDON, "Version") or ""; 

local ADDON, VERSION 	= gLib.ADDON, gLib.VERSION;

local Addon 			= {};
local Event 			= {};
local L 				= {};
local lib 				= {};
local Tags 				= {};
local Task 				= {};

local bagstrings = {
	["backpack"]		= { 0 },
	["bags"]			= { 1, 2, 3, 4 },
	["backpack+bags"]	= { 0, 1, 2, 3, 4, },
	["bank"]			= { 5, 6, 7, 8, 9, 10, 11 },
	["bankframe"]		= { -1 },
	["bankframe+bank"]	= { -1, 5, 6, 7, 8, 9, 10, 11 },
	["keyring"]			= { -2 },
};

local err_message = {
	["ERR_USAGE"] 								= "Usage: %s";
	["ERR_NAME_REQUIRED"] 						= "'name' required";
	["ERR_ALREADY_EXISTS"] 						= "'%s' already exists";
};

local LOCALES = { 
	deDE = true; 
	enUS = true; 
	frFR = true; 
	koKR = true; 
	ruRU = true; 
	zhCN = true; 
	zhTW = true; 
};

local ReservedEvents = { 
	ADDON_LOADED 		= true; 
	PLAYER_LOGIN 		= true; 
	PLAYER_LOGOUT 		= true;
};

local VALIDPOINTS = { 
	BOTTOM 			= true; 
	BOTTOMLEFT 		= true; 
	BOTTOMRIGHT 	= true; 
	CENTER 			= true; 
	LEFT 			= true; 
	RIGHT 			= true; 
	TOP 			= true; 
	TOPLEFT 		= true; 
	TOPRIGHT 		= true; 
};

local _G = _G;
local abs, date, error, floor, format, gsub, mod, setmetatable, sort, strlen, strsub, strupper, tinsert, tremove, tonumber, tostring, type, unpack = abs, date, error, floor, format, gsub, mod, setmetatable, sort, strlen, strsub, strupper, tinsert, tremove, tonumber, tostring, type, unpack;

self = gLib;
self.Frame, self.Menu, self.Object, self.String, self.System = lib, lib, lib, lib, lib;

------------------------------------------------------------------------------------------------------------
-- 	Main Frame
------------------------------------------------------------------------------------------------------------

local frame = CreateFrame("Frame");

for event, reg in pairs(ReservedEvents) do
	if reg then 
		frame:RegisterEvent(event);
	end
end

-- credits to Lonny and Paul Emmerich for inspiring this
local function OnUpdate(self, elapsed)
	for addon, tasks in pairs(Task) do
		for i,v in pairs(tasks) do
			v.elapsed = (v.elapsed or 0) + elapsed
			if (v.elapsed >= v.interval) then
				v.func(_G[addon]);
				v.elapsed = v.elapsed - v.interval;
			end
		end
	end
end
local function OnEvent(self, event, ...)
	if ReservedEvents[event] then
		frame[event](self, event, ...);
		return
	end
	if Event[event] then
		for addon, func in pairs(Event[event]) do
			if (type(func) == "string") then
				_G[addon][event](_G[addon], event, ...)
			elseif (type(func) == "function") then
				func(_G[addon], event, ...)
			end
		end
	end
end

--
-- Call's all the registered addons' OnInit\OnEnable\OnDisable methods
-- 	Will also call ADDON_LOADED\PLAYER_LOGIN\PLAYER_LOGOUT if they exists (for the sake of backwards compability)
--
function frame:ADDON_LOADED(event, name)
	if Addon[name] then
		Addon[name]:OnInit();
		
		-- backwards compability
		if Addon[name].ADDON_LOADED then
			Addon[name]:ADDON_LOADED();
		end
	end
end
function frame:PLAYER_LOGIN(self, event)
	for name, addon in pairs(Addon) do
		addon:OnEnable();

		-- backwards compability
		if addon.PLAYER_LOGIN then
			addon:PLAYER_LOGIN();
		end
	end
end
function frame:PLAYER_LOGOUT(self, event)
	for name, addon in pairs(Addon) do
		addon:OnDisable();

		-- backwards compability
		if addon.PLAYER_LOGOUT then
			addon:PLAYER_LOGOUT();
		end
	end
end

frame:SetScript("OnUpdate", OnUpdate);
frame:SetScript("OnEvent", OnEvent);


------------------------------------------------------------------------------------------------------------
-- 	Template Addon
------------------------------------------------------------------------------------------------------------

local prototypeAddon = {};

function prototypeAddon:OnInit()
	return
end
function prototypeAddon:OnEnable()
	return
end
function prototypeAddon:OnDisable()
	return
end

--
-- 	usage:	myAddon:RegisterEvent( event[, func] )
-- 			- Same as the normal API RegisterEvent
-- 			- Registers the event with gLib's handler
--
function prototypeAddon:RegisterEvent(event, func)
	if (type(event) ~= "string") or (event == "") then return end
	if ReservedEvents[event] then return end

	if not Event[event] then
		Event[event] = {};
		frame:RegisterEvent(event)
	end
	
	Event[event][self:GetName()] = func or event
end

--
-- 	usage:	myAddon:RegisterEvents( { "EVENT1", ["EVENT2"} = function() end, EVENT3 = function() end, "EVENT4", ...etc } )
-- 			- Registers multiple events at once
-- 			- If the value is a function, then the index must be the eventname
-- 			- If the value is a string, the value will be the event name, and the myAddon:EVENTNAME() will be called when the event fire
--
function prototypeAddon:RegisterEvents(events)
	if (type(events) ~= "table") then return end
	
	for i,v in pairs(events) do
		if type(v) == "string" then
			self:RegisterEvent(v);
		elseif type(v) == "function" then
			self:RegisterEvent(i, v);
		end
	end
end
function prototypeAddon:UnregisterEvent(event)
	if (type(event) ~= "string") or (event == "") then return end
	if ReservedEvents[event] then return end
	
	if Event[event][self] then
		Event[event][self] = nil
	end
	if #Event[event] == 0 then
		Event[event] = nil;
		frame:UnregisterEvent(event);
	end
end

--
-- 	usage: 	myAddon:Schedule( task[, interval] )
-- 			- registers the function <task> to be called every <interval> seconds
-- 			- interval must be between 0.1s and 999s, or it will default to 0.5s
--
function prototypeAddon:Schedule( task, interval )
	if (type(task) ~= "function") then return end
	
	Task[self] = Task[self] or {};
	
	local thetask = {
		func = task;
		interval = min(max(((type(interval) == "number") and abs(interval) or 0.5), 0.1), 999);
	};
	
	tinsert(Task[self], thetask)
	
end

-- 
-- previously all addon objects were frames, now they are tables
-- 	GetName() now returns the addons name
--
function prototypeAddon:GetName()
	return self.ADDON or gLib.ADDON
end

--
-- create a new addon (inherits all methods from prototypeAddon)
--
function gLib:newAddon( name )
	if not(name) then return end
	if (type(name) ~= "string") then return end
	if (name == "") then return end
	if _G[name] then return end

	Addon[name] = gLib:newClass( name, prototypeAddon );
	Addon[name].newClass = gLib.newClass;
	
	lib:SetMeta( Addon[name], name );
	lib:Embed( Addon[name], lib);

	_G[name] = Addon[name];

	return Addon[name]
end

--
-- create a new instance of the class parentClass
-- that inherits all of parentClass' methods
--
-- 	newClass = gLib:newClass( "myClass", yourClass )
-- 	
function gLib:newClass( name, parentClass )

    local newClass = {};
    local class_mt = { __index = newClass };

    function newClass:create()
        local newinst = {};
        setmetatable( newinst, class_mt );
        return newinst
    end

    if parentClass then
        setmetatable( newClass, { __index = parentClass } );
    end

    -- Return the class object of the instance
    function newClass:class()
        return newClass
    end

    -- Return the super class object of the instance
    function newClass:superClass()
        return parentClass
    end

    -- Return true if the caller is an instance of theClass
    function newClass:isa( theClass )
        local b_isa = false;

        local cur_class = newClass;

        while ( nil ~= cur_class ) and ( false == b_isa ) do
            if cur_class == theClass then
                b_isa = true;
            else
                cur_class = cur_class:superClass();
            end
        end

        return b_isa
    end
	
    return newClass
	
end

--
-- 	if no locale is given (nil), the content is passed to all locales
--
function gLib:AddLocale( addon, locale, content )
	if not addon then return end
	
	-- create addon's main locale table 
	L[addon] = L[addon] or {}

	-- add the content to a specific locale if given
	if LOCALES[locale] then
		-- create the locale table
		L[addon][locale] = L[addon][locale] or {}
	
		-- if no content was given, we're done here
		if not content then return end

		-- add the content
		lib:Copy( content, L[addon][locale] );
		
	-- add the content to ALL locales
	else
		for i,v in pairs(LOCALES) do
			-- add the locale
			L[addon][i] = L[addon][i] or {}

			-- add the content, if any
			if content then
				lib:Copy( content, L[addon][i] );
			end
		end
	end
	
end

--
-- 	if no locale is given (nil), the client's default locale is returned
--
function gLib:FetchLocale( addon, locale )
	if not (L[addon]) then return end
	if locale and not(L[addon][locale]) then return end

	return L[addon][locale or GetLocale()]
end

-- all-in-one function
-- allows you to create an addon, register events, setup slash commands, and validate settings
function gLib:New( db )
	db = lib:ArgumentCheck( db, { name = ""; events = {}; slash = {}; userDB = {}; defaultDB = {}; statics = {}; });

	if (db.name == "") or (_G[db.name]) then return end

	local addon = gLib:newAddon( db.name );
	
	addon:RegisterEvents(db.events);

	for i,v in pairs(db.slash) do
		addon:Slash( db.name.."_"..db.slash[i].name, db.slash[i].func, db.slash[i].slashes );
	end

	addon.defaults = db.defaultDB;
	addon.statics = db.statics;
	
	return addon, addon:ArgumentCheck( db.userDB, db.defaultDB )
end

-- debugging
-- prints a list of currently handled gLib addons/objects
function gLib:list()
	for name,addon in pairs(Addon) do
		print( ("%s v%s - %s"):format(name, addon.VERSION, addon.NAME) );
	end
end




-- compare the types of the user input to the default settings, 
-- and fill in the holes with the given defaults
function lib:ArgumentCheck( user, defaults ) 
	if not(defaults) then return end
	if (not(user)) or (type(user) ~= "table") then 
		user = {}; 
	end
	for i,v in pairs(defaults) do
		-- get rid of nil values, keep false
		user[i] = ((user[i] == nil) and self:Copy(defaults[i])) or user[i];
		if type(user[i]) ~= type(defaults[i]) then
			-- if the type is wrong, replace with default value
			user[i] = self:Copy(defaults[i]);
		end
	end
	return user
end

-- crop a string with trailing ...
function lib:Crop( str, maxlength )
	if not(str) or not(type(str) == "string") or (string.len(str) == 0) then return end
	maxlength = (maxlength < 4) and 4 or maxlength
	if strlen(str) <= maxlength then return str end
	return strsub(str, 1, maxlength-3 ).."..."
end

-- return only the uppercase characters from a given string
function lib:Upper( str )
	-- usual syntax checks	
	if not(str) then return end
	if type(str) ~= "string" then return end
	if strlen(str) == 0 then return end
	
	-- return the string stripped of lowercase characters
	return (gsub(str, "[%l]", ""))
end

-- return only the lowercase characters from a given string
function lib:Lower( str )
	-- usual syntax checks	
	if not(str) then return end
	if type(str) ~= "string" then return end
	if strlen(str) == 0 then return end
	
	-- return the string stripped of uppercase characters
	return (gsub(str, "[%u]", ""))
end




------------------------------------------------------------------------------------------------------------
-- 	API Calls
------------------------------------------------------------------------------------------------------------

local function Size(obj, w, h)
	if not(tonumber(h)) then h = w; end

	obj:SetWidth(w);
	obj:SetHeight(h);
end
local function Place(obj, arg1, arg2, arg3, arg4, arg5)
	obj:ClearAllPoints();
	obj:SetPoint(arg1, arg2, arg3, arg4, arg5);
end

-- credits to Nightcracker for the basic idea here
local function apply(object)
	getmetatable(object).__index.Size = Size
	getmetatable(object).__index.Place = Place
end

apply(CreateFrame("Frame"):CreateTexture())
apply(CreateFrame("Frame"):CreateFontString())

local handled, object = {}, EnumerateFrames()
while object do
	if not handled[object:GetObjectType()] then
		apply(object)
		handled[object:GetObjectType()] = true
	end

	object = EnumerateFrames(object)
--	if not object then break end
end


--
-- 	usage: 	{substr1, substr2,...} = string.split(str, separator) or str:split(separator)
--
local function split(str, separator)
	local t = {}  
	local fpat = "(.-)" .. separator
	local last_end = 1
	local s, e, cap = str:find(fpat, 1)
	while s do
		if s ~= 1 or cap ~= "" then
			tinsert(t,cap)
		end
		last_end = e+1
		s, e, cap = str:find(fpat, last_end)
	end
	if last_end <= #str then
		cap = str:sub(last_end)
		tinsert(t, cap)
	end
	return t
end

--
-- 	usage: 	string.tag("[tagname:arg1,arg2,arg3...] some text [tagname:arg1,arg2,arg3...]") or str:tag()
--
-- credits to cargor for the inspiration to this
--
function lib:Tag( str  )
	if not(str) or (type(str) ~= "string") then return end

--	return (str:gsub("%[([^%]:]+):?(.-)%]", function(a,b) return tagger(a, split(b, "[,]+")) end))
	return (str:gsub("%[([^%]:]+):?(.-)%]", function(a,b) return Tags[a](unpack(split(b, "[,]+"))) end))
end
string.tag = function(tagstr) return lib:Tag(tagstr) end

function lib:RegisterTag( tag, func )
	if (type(tag) ~= "string") then return end
	if (type(func) ~= "function") then return end
	if Tags[tag] then return end
	
	Tags[tag] = func;
end
string.addtag = function(tag, func) return lib:RegisterTag(tag, func) end


------------------------------------------------------------------------------------------------------------
-- 	Tags
------------------------------------------------------------------------------------------------------------

--
-- 	displays player money if no arguments are given, otherwise shows the money given
-- 		any value of <full> that doesn't return nil or false will show the full coinstring
-- 		if <full> is omitted, or returns nil or false, the coinstring will be truncated to ignore empty fields
-- 		if the total money value is 0, it will display as 0 copper
--
Tags["money"] = function(money, full)
	money = money and tonumber(money) or GetMoney();

	local str;
	local g,s,c = floor(money/1e4), floor(money/100) % 100, money % 100;
	local pos = "0:0:2:0"

	if full then
		if(g > 0) then str = (str and str.." " or "") .. g .. lib:CreateIcon("Interface\\MoneyFrame\\UI-GoldIcon", pos) end
		if(s > 0) or (g > 0) then str = (str and str.." " or "") .. s .. lib:CreateIcon("Interface\\MoneyFrame\\UI-SilverIcon", pos) end
		str = (str and str.." " or "") .. c .. lib:CreateIcon("Interface\\MoneyFrame\\UI-CopperIcon", pos)
		return str
	else
		if(g > 0) then str = (str and str.." " or "") .. g .. lib:CreateIcon("Interface\\MoneyFrame\\UI-GoldIcon", pos) end
		if(s > 0) then str = (str and str.." " or "") .. s .. lib:CreateIcon("Interface\\MoneyFrame\\UI-SilverIcon", pos) end
		if(c > 0) or ( (g + s + c) == 0 ) then str = (str and str.." " or "") .. c .. lib:CreateIcon("Interface\\MoneyFrame\\UI-CopperIcon", pos) end
		return str
	end
end

local function GetBagSpace(bags)
	if not bagstrings[bags] then return end

	local free, total, used = 0, 0, 0;
	for _,i in pairs( bagstrings[bags] ) do
		free, total, used = free + GetContainerNumFreeSlots(i), total + GetContainerNumSlots(i), total - free;
	end
	return free, total, used, floor(free / total * 100)
end

-- 
-- 	bagspace
--  		bags can be;
-- 			backpack 			- Only the backpack
-- 			bags 				- All the character's equipped bags EXCEPT the backpack
-- 			backpack + bags		- All the character's equipped bags and the backpack
-- 			bankframe 			- Only the bankframe
-- 			bank 				- All the character's equipped bank bags EXCEPT the bankframe
-- 			bankframe + bank		- All the character's equipped bank bags and the bankframe
-- 			keyring 				- The keyring
--
Tags["free"] = function(bags)
	if not bagstrings[bags] then return end
	return (select(1, GetBagSpace(bags)))
end
Tags["max"] = function(bags)
	if not bagstrings[bags] then return end
	return (select(2, GetBagSpace(bags)))
end
Tags["used"] = function(bags)
	if not bagstrings[bags] then return end
	return (select(3, GetBagSpace(bags)))
end
Tags["freepercent"] = function(bags)
	if not bagstrings[bags] then return end
	return (select(4, GetBagSpace(bags))).."%"
end
Tags["usedpercent"] = function(bags)
	if not bagstrings[bags] then return end
	return (100-(select(4, GetBagSpace(bags)))).."%"
end

--
-- 	time
-- 		timezone can be; 
-- 			game24/game (or nil) 	- 24 hour server/game time
-- 			game12 		 		- 12 hour server/game time
-- 			local24/local 		- 24 hour local time
--			local12 				- 12 hour local time
-- 			pvp 					- time until pvp wears off
-- 			wintergrasp 			- time until next Wintergrasp
--
Tags["time"] = function( timezone )
	timezone = timezone and timezone:lower() or "game"

	if ((timezone == "local") or (timezone == "local24")) then
		return date("%H:%M")

	elseif (timezone == "local12") then
		return date("%I:%M %p")

	elseif (timezone == "game") or (timezone == "game24") then
		local hours, minutes = GetGameTime();
		return (format("%02d", hours)..":"..format("%02d", minutes))

	elseif (timezone == "game12") then
		local hours, minutes = GetGameTime();
		return (format("%02d", ((hours > 12) or (hours < 1)) and (hours - 12) or hours)..":"..format("%02d", minutes)..((hours > 12) and " PM" or " AM"));

	elseif (timezone == "pvp") then
		local sec = GetPVPTimer()
		if (sec >= 0) and (sec <= 301000) then
			seconds = floor(seconds/1000);
			return (format("%d", floor(seconds/60) % 60)..":"..format("%02d", seconds % 60))
		else
			return "--:--"
		end

	elseif (timezone == "wintergrasp") then
		local seconds = GetWintergraspWaitTime();
		if seconds then
			return (format("%d", floor(seconds/3600))..":"..format("%02d", floor(seconds/60) % 60)..":"..format("%02d", seconds % 60))
		else
			return ("--:--:--")
		end
	end
end

--
-- 	combat stats
--		ap 					- current melee attack power 
--		rap 					- current ranged attack power 
--
Tags["ap"] = function(unit)
	local base, posBuff, negBuff = UnitAttackPower(unit or "player");
	return (base + posBuff + negBuff)
end
Tags["rap"] = function(unit)
	local base, posBuff, negBuff = UnitRangedAttackPower(unit or "player");
	return (base + posBuff + negBuff)
end


------------------------------------------------------------------------------------------------------------
--
--  Even though embedded into new addons, avoid calling the following:
--
------------------------------------------------------------------------------------------------------------

-- old routines
function lib:noop()
	return
end
function lib:Cvars( cvars )
	for i,v in pairs(cvars) do
		SetCVar(i, v);
	end
end

-- system output, with an optional header
function lib:Print( header, ... )
	-- set output header
	header = header or ""
	
	--[[
		Prints out any number of system messages 
		on separated lines in the main chat frame.
		
		{...} can be any number of comma-separated messages, 
		or a table containing strings. 
		
		Empty values "" will add a blank line with no header message.
	]]--
	
	local args = {...};
	function printmessage(args)
		for i, a in pairs(args) do
			-- call itself if the input string is a table
			if type(a) == "table" then 
				printmessage(a) 
			else
				DEFAULT_CHAT_FRAME:AddMessage(((a ~= "" ) and header or " ")..a);
			end
		end
	end
	printmessage(args);
end

-- creates slashcommands
-- 		name 		= internal name
-- 		func 		= function to call
-- 		slashes 		= table of slash commands without the /
function lib:Slash( name, func, slashes)
	-- validate input
	if not(name) or type(name)~= "string" then return end
	func = ((type(func) == "function") and func) or (function() end)

	-- create slash commands
	for i,v in pairs(slashes) do
		_G["SlashCmdList"][name] = function(...) func(...) end;
		_G["SLASH_"..name..i] = "/"..v;
	end
end

-- sets certain values for the addon
function lib:SetMeta( object, addon )
	object.ADDON 			= addon; 
	object.VERSION 			= GetAddOnMetadata(object.ADDON, "Version") or ""; 
	object.NAME 			= GetAddOnMetadata(object.ADDON, "Title") or addon; 
	object.REALM 			= (GetRealmName());
	object.CHARNAME 		= (select(1, UnitName("player")));
	object.FACTION 			= (UnitFactionGroup("player"));
end

-- icon
function lib:CreateIcon(icon, iconValues)
	iconValues = iconValues or "0:0:0:0";
	if(type(iconValues) == "table") then
		iconValues = table.concat(iconValues, ":")
	end
	return ("|T%s:%s|t"):format(icon, iconValues)
end


-- shows a smart time string
function lib:Time(seconds)
	if not(seconds) then return "0.0"; end
	local day, hour, minute = 86400, 3600, 60
	if seconds >= day then
		return format("%dd", floor(seconds/day + 0.5)), seconds % day
	elseif seconds >= hour then
		return format("%dh", floor(seconds/hour + 0.5)), seconds % hour
	elseif seconds >= minute then
		if seconds <= minute * 5 then
			return format("%d:%02d", floor(seconds/60), seconds % minute), seconds - floor(seconds)
		end
		return format("%dm", floor(seconds/minute + 0.5)), seconds % minute
	elseif seconds >= minute / 12 then
		return floor(seconds + 0.5), (seconds * 100 - floor(seconds * 100))/100
	end
	return format("%.1f", seconds), (seconds * 100 - floor(seconds * 100))/100
end

-- shows a smart abbreviated value
function lib:Value(value)
	if not(value) then return "0.0"; end
	if value >= 1e6 then
		return ("%.1fm"):format(value / 1e6):gsub("%.?0+([km])$", "%1")
	elseif value >= 1e3 or value <= -1e3 then
		return ("%.1fk"):format(value / 1e3):gsub("%.?0+([km])$", "%1")
	else
		return value
	end
end

-- embeds all the values and methods from source into target
function lib:Embed( target, source )
	if not target or not source then return end
	for i,v in pairs(source) do
--		target[i] = source[i];
		target[i] = v;
	end
end

-- fully copy a (nested) table, instead of simply creating pointers
function lib:Copy( old, new )
	-- if old is not a table, then just return it
	if type(old) ~= "table" then return old; end 

	-- smart function, hell yeah
	local function iterate(oldt, newt)
		for i,v in pairs(oldt) do
			if type(oldt[i]) == "table" then 
				newt[i] = {};
				iterate(oldt[i], newt[i]);
			else
				newt[i] = oldt[i];
			end
		end
	end

	-- create our new table if it doesn't exist already
	local new = ((new and type(new) == "table") and new) or {}
	iterate(old, new);

	-- return the new or modified table
	return new
end

-- create a table with all given values set to true
--  useful for syntax checks, localization, etc
function lib:TrueTable( indexlist )
	if not indexlist then return nil end
	if not(type(indexlist) == "table") then return nil end
	local truetable = {}
	for i,v in pairs(indexlist) do
		-- take the values from our original table, 
		-- and turn them into indexes in our new.
		truetable[v] = true;
	end
	return truetable
end


-- reset a user database to it's defaults
function lib:SetDefaults( user, defaults )
	if not(defaults) then return end
	if (not(user)) or (type(user) ~= "table") then 
		user = {}; 
	end
	user = self:Copy( defaults, user )
	return user
end 

-- create a sorted array of the keys in a given table
function lib:SortByKey( tbl )
	if not tbl then return end
	if not(type(tbl)=="table") then return end
	
	-- create a new empty table to sort our keys in
	local ntbl = {}
	
	for i,v in pairs(tbl) do
		tinsert( ntbl, i ) 
	end
	
	-- sort our new table
	sort( ntbl )
	
	-- return our sorted array
	return ntbl
end

-- sets the visibily of object
function lib:Toggle( object, state )
	-- Hides/Shows any given element
	if state then
		object:Show();
	elseif not state then
		object:Hide();
	end
end

-- get or set the position of a frame, with a few fail-safes
function lib:GetPos( frame )
	local point, anchor, relpoint, x, y = frame:GetPoint();
	return { 
		point = point;
		anchor = "UIParent"; 
		relpoint = relpoint; 
		x = tonumber(x);
		y = tonumber(y);
	}
end
function lib:SetPos( frame, where )
	where = where or {};

	local point 	= self.VALIDPOINTS[where.point] and tostring(where.point) or "TOPLEFT";
	local anchor 	= "UIParent";
	local relpoint 	= self.VALIDPOINTS[where.relpoint] and tostring(where.relpoint) or "TOPLEFT";
	local x 		= tonumber(where.x) or 0;
	local y 		= tonumber(where.x) or 0;

	frame:SetPoint(point, anchor, relpoint, x, y);
end



------------------------------------------------------------------------------------------------------------
--
-- 	Interface Options Menus
--
------------------------------------------------------------------------------------------------------------

-- Avoid calling these before the PLAYER_LOGIN() Event has fired
--[[

	Converts the array given in {panel} to an Interface Options menu,
		and returns the link to the new menu object.
	
	All the .name values are valid as links for the InterfaceOptionsFrame_OpenToCategory()
		function to open to.
	
	Format:

		panel.title 			= title of the addon to which the menu belongs
		panel.version 		= version number of the addon
		panel.description 	= description of the addon, the text shown on the menu's front page
		
		panel.okay 			= Function to be executed when you hit "Ok"
		panel.cancel 			= Function to be executed when you hit "Cancel"
		panel.default 		= Function to be executed when you hit "Default"
		panel.refresh 		= Hell if I know
		
		panel.option[x] = frontpage menu options array, se below for more settings
			panel.option[x].name 			= text on the menu option
			panel.option[x].type 			= what kind of menu option?
											"button" = normal checkbutton
											"dropdown" = dropdown menu										
											"editbox" = editbox											TODO
											"slider" = sliderbar											TODO
											"colorpicker" = function that provides an inputbox with			TODO
												a hexadecimal color value, as well as a colorwheel
												accessible by clicking the color thumbnail 
			panel.option[x].list 			= dropdown menus												
			panel.option[x].password 		= boolean. if true, editbox will mask the contents					TODO
			panel.option[x].min 			= minimum value assigned to sliders									TODO
			panel.option[x].max 			= maximum value assigned to sliders									TODO
			panel.option[x].maxletters 		= sets the maximum numbers of allowed letters in an editbox 			TODO
			panel.option[x].value 			= value of the menu option
			panel.option[x].panelfunction 	= function connected to the menu option
		
		panel.menu[x] = submenu
			panel.menu[x].name = title of the subtree
			panel.menu[x].option[x] = frontpage menu options array, se below for more settings
				panel.menu[x].option[x].name 			= text on the menu option
				panel.menu[x].option[x].type 			= what kind of menu option?
														"button" = normal checkbutton
														"dropdown" = dropdown menu										
														"editbox" = editbox											TODO
														"slider" = sliderbar											TODO
														"colorpicker" = function that provides an inputbox with			TODO
													a hexadecimal color value, as well as a colorwheel
													accessible by clicking the color thumbnail 
				panel.menu[x].option[x].list 			= dropdown menus
				panel.menu[x].option[x].password 		= boolean. if true, editbox will mask the contents					TODO
				panel.menu[x].option[x].min 			= minimum value assigned to sliders									TODO
				panel.menu[x].option[x].max 			= maximum value assigned to sliders									TODO
				panel.menu[x].option[x].maxletters 		= sets the maximum numbers of allowed letters in an editbox 			TODO
				panel.menu[x].option[x].value 			= value of the menu option
				panel.menu[x].option[x].panelfunction 	= function connected to the menu option
	
	Example:
		your_panel = Menu.CreateOptionsPanel(your_array)
		InterfaceOptionsFrame_OpenToCategory(your_array.menu[3].name)
		
		* 	Creates an Interface Options menu tree based on the table your_array, 
			and opens the menu at the 4th submenu.
]]--
function lib:CreateOptionsPanel( panel, userDB, defaultDB )
	if not panel then return end
	if not(panel.title) then return end
	
	local p
	
	-- is this a new menu, or does it belong to an existing one?
	if not panel.panel then
		p = CreateFrame("Frame", "Menu_"..panel.title, UIParent);

		-- create main container frame for the menu tree
		p.head = self:CreateText({
			anchor 			= p; 
			framestrata 	= "ARTWORK";
			font 			= (select(1, GameFontNormalLarge:GetFont())); 
			fontsize 		= (select(2, GameFontNormalLarge:GetFont()));
			name 			= panel.title.."OPTIONS_TITLE";
			offsetX 		= 14;
			offsetY 		= -14;
			parent 			= p;
			text 			= panel.title.." "..panel.version;
		});
		p.name = panel.title.." "..panel.version;
		if panel.description then 
			p.body = self:CreateText({
				parent 			= p;
				name 			= panel.title.."OPTIONS_DESCRIPTION"; 
				framestrata 	= "ARTWORK";
				font 			= (select(1, GameFontWhite:GetFont()));
				fontsize 		= (select(2, GameFontWhite:GetFont()));			
				anchor 			= p.head;
				offsetX 		= 8;
				offsetY 		= -20;
				text 			= panel.description;
			});
		end
	else
		-- hook this up to another menu?
		p = CreateFrame("Frame", "Menu_"..panel.title, panel.panel);
		
		p.name = panel.title.." "..panel.version; 
		p.head = self:CreateText({
			anchor 			= p; 
			framestrata 	= "ARTWORK";
			font 			= (select(1, GameFontNormalLarge:GetFont()));
			fontsize 		= (select(2, GameFontNormalLarge:GetFont()));
			name 			= p.name.."_TITLE";
			offsetX 		= 14;
			offsetY 		= -14;
			parent 			= p;
			text 			=  panel.title.." "..panel.version;
		});
		if panel.description then 
			p.body = self:CreateText({
				parent 			= p;
				name 			= panel.title.."OPTIONS_DESCRIPTION"; 
				framestrata 	= "ARTWORK";
				font 			= (select(1, GameFontWhite:GetFont()));
				fontsize 		= (select(2, GameFontWhite:GetFont()));			
				anchor 			= p.head;
				offsetX 		= 8;
				offsetY 		= -20;
				text 			= panel.description;
			});
		end
		
		p.parent = panel.panel.name
	end
	
	-- if it exists, show the frontpage's menu options
	if panel.option then
		p.option = {};
		
		local previousparent = p.head;
		local xoff,yoff = 0, -28 ; 
		
		-- get a sorted array of the keys in panel.option
		local stbl = self:SortByKey(panel.option);
		
		-- iterate over our sorted array, retrieve the keys for panel.option in order
		for stblcount = 1, #stbl,1 do
			local maincounter = stbl[stblcount] 
			if maincounter == 0 and p.body then yoff = yoff - p.body:GetHeight(); end;
			if panel.option[maincounter].type == "button" then
				p.option[maincounter] = self:CreateCheckButton({
					name 				= "Menu_"..p.name.."_CB"..maincounter; 
					text 				= panel.option[maincounter].name;
					parent 				= p;
					anchor 				= previousparent; 
					offsetX 			= xoff; 
					offsetY 			= yoff;
					checked 			= panel.option[maincounter].value;
					onclick 			= panel.option[maincounter].panelfunction;
				});
			end
			if panel.option[maincounter].type == "dropdown" then
				p.option[maincounter] = self:CreateDropDown({
					name 				= "Menu_"..p.name.."_CB"..maincounter; 
					text 				= panel.option[maincounter].name;
					parent 				= p;
					anchor 				= previousparent; 
					offsetX 			= xoff; 
					offsetY 			= yoff;
					menutable 			= panel.option[maincounter].list;
				});
			end
			yoff = -(p.option[maincounter]:GetHeight() + 2);
			previousparent = p.option[maincounter];
		end
		
		-- set the parent to our menu
		p.option.parent = (panel.panel and panel.panel.name) or p.name;
	end

	-- setup the control functions
	p.okay = panel.okay or function () end;
	p.cancel = panel.cancel or function () end;
	p.default = function()
		if not(userDB) or not(defaultDB) then return end
		
		-- overwrite all options with the defaults
		userDB = self:SetDefaults( userDB, defaultDB );
		
		-- call the user's function, if any
		if panel.default then panel.default(); end
		
		-- This doesn't appear to reload the UI, just the menu. Perfect.
		ReloadUI();
		
	end;
	p.refresh = panel.refresh or function () end

	-- add main menu object
	InterfaceOptions_AddCategory(p);

	
	-- if we're hooking this menu into an existing one, we don't really want submenus
	if not panel.panel then
		-- create submenus
		if panel.menu then
			p.menu = {};
			
			-- get a sorted array of the keys in panel.option
			local stbl = self:SortByKey(panel.menu);
			
			-- iterate over our sorted array, retrieve the keys for panel.option in order
			for stblcount = 1, #stbl,1 do
				local counter = stbl[stblcount] 
				p.menu[counter] = CreateFrame("Frame", panel.menu[counter].name, p);
				p.menu[counter].name = panel.menu[counter].name; 
				p.menu[counter].head = self:CreateText({
					anchor 			= p.menu[counter]; 
					framestrata 	= "ARTWORK";
					font 			= (select(1, GameFontNormalLarge:GetFont()));
					fontsize 		= (select(2, GameFontNormalLarge:GetFont()));
					name 			= p.menu[counter].name.."_TITLE";
					offsetX 		= 14;
					offsetY 		= -14;
					parent 			= p.menu[counter];
					text 			= p.name.." "..panel.menu[counter].name;
				});
				if panel.menu[counter].description then 
					p.menu[counter].body = self:CreateText({
						parent 			= p.menu[counter];
						name 			= panel.menu[counter].name.."SUBOPTIONS_DESCRIPTION"..counter; 
						framestrata 	= "ARTWORK";
						font 			= (select(1, GameFontWhite:GetFont()));
						fontsize 		= (select(2, GameFontWhite:GetFont()));
						anchor 			= p.menu[counter].head;
						offsetX 		= 8;
						offsetY 		= -20;
						text 			= panel.menu[counter].description;
					});
				end
				p.menu[counter].option = {};
				local previousparent = p.menu[counter].head;
				local xoff,yoff = 0, -28;

				-- get a sorted array of the keys in panel.option
				local sstbl = self:SortByKey(panel.menu[counter].option);
				
				-- iterate over our sorted array, retrieve the keys for panel.option in order
				for sstblcount = 1, #sstbl,1 do
					local subcounter = sstbl[sstblcount] 
					if subcounter == 0 and p.menu[counter].body then yoff = yoff - p.menu[counter].body:GetHeight(); end				
					if panel.menu[counter].option[subcounter].type == "button" then
						p.menu[counter].option[subcounter] = self:CreateCheckButton({
							name 				= "Menu_"..p.menu[counter].name.."_CB"..subcounter; 
							text 				= panel.menu[counter].option[subcounter].name;
							parent 				= p.menu[counter];
							anchor 				= previousparent; 
							offsetX 			= xoff; 
							offsetY 			= yoff;
							checked 			= panel.menu[counter].option[subcounter].value;
							onclick 			= panel.menu[counter].option[subcounter].panelfunction;
						});
					end
					if panel.menu[counter].option[subcounter].type == "dropdown" then
						p.menu[counter].option[subcounter] = self:CreateDropDown({
							name 				= "Menu_"..p.menu[counter].name.."_CB"..subcounter; 
							text 				= panel.menu[counter].option[subcounter].name;
							parent 				= p.menu[counter];
							anchor 				= previousparent; 
							offsetX 			= xoff; 
							offsetY 			= yoff;
							menutable 			= panel.option[maincounter].option[subcounter].list;
						});
					end
					previousparent = p.menu[counter].option[subcounter];
					yoff = -(p.menu[counter].option[subcounter]:GetHeight() + 2);
				end
				
				-- add our submenu				
				p.menu[counter].parent = p.name;
				InterfaceOptions_AddCategory(p.menu[counter]);
			end
		end
	end
	
	-- return the menu object
	return p
end

function lib:CreateCheckButton( args )
	args = self:ArgumentCheck( args, {
		name 						= "";
		text 						= "";
		maxwidth 					= InterfaceOptionsFramePanelContainer:GetWidth() - 40;
		parent 						= UIParent;
		relpoint 					= "TOPLEFT";
		anchor 						= UIParent;
		offsetX 					= 0;
		offsetY 					= 0;
		checked 					= false;
		justifyH 					= "LEFT";
		justifyV 					= "TOP"; 
		onclick						= function() end;
	});
	
	-- double check region points and alignments
	if not(VALIDPOINTS[args.relpoint]) then args.relpoint = "TOPLEFT"; end
	if not((args.justifyH == "RIGHT") or (args.justifyH == "LEFT")) then args.justifyH = "LEFT"; end
	if not((args.justifyV == "TOP") or (args.justifyV == "BOTTOM")) then args.justifyV = "TOP"; end

	local b = CreateFrame("CheckButton", args.name, args.parent, "InterfaceOptionsCheckButtonTemplate");
	b:SetPoint( args.relpoint, args.anchor, args.relpoint, args.offsetX, args.offsetY );
	b:SetChecked( args.checked );
	b:SetScript( "OnClick", args.onclick );

	b.text = self:CreateText({
		parent 				= b; 
		name 				= args.name.."Text";
		framestrata 		= "ARTWORK";
		font 				= (select(1, GameFontWhite:GetFont())); 
		fontsize 			= (select(2, GameFontWhite:GetFont())); 
		point 				= "LEFT";
		relpoint 			= "RIGHT";
		anchor 				= b;
		text 				= args.text;
		maxwidth 			= args.maxwidth - b:GetWidth();
	});

	return b;
end

function lib:CreateText( args )
	args = self:ArgumentCheck( args, {
		parent						= UIParent; 
		name 						= "";
		framestrata 				= "BACKGROUND"; 
		font 						= (select(1, GameFontNormal:GetFont())); 
		fontsize 					= (select(2, GameFontNormal:GetFont())); 
		point 						= "TOPLEFT"; 
		relpoint 					= "TOPLEFT"; 
		anchor 						= UIParent; 
		offsetX 					= 0; 
		offsetY 					= 0; 
		justifyH 					= "LEFT"; 
		justifyV 					= "TOP"; 
		text 						= ""; 
		maxwidth 					= InterfaceOptionsFramePanelContainer:GetWidth() - 40; 
	});
	
	--double check points and alignments
	if not(VALIDPOINTS[args.point]) then args.point = "TOPLEFT"; end
	if not(VALIDPOINTS[args.relpoint]) then args.relpoint = "TOPLEFT"; end
	if not((args.justifyH == "RIGHT") or (args.justifyH == "LEFT")) then args.justifyH = "LEFT"; end
	if not((args.justifyV == "TOP") or (args.justifyV == "BOTTOM")) then args.justifyV = "TOP"; end

	local f = args.parent:CreateFontString( args.name, args.framestrata );
	f:SetWordWrap( true );
	f:SetNonSpaceWrap( false );
	f:SetWidth( args.maxwidth );
	f:SetFont( args.font, args.fontsize );
	f:SetText( args.text );
	f:SetJustifyH( args.justifyH );
	f:SetPoint( args.point, args.anchor, args.relpoint, args.offsetX, args.offsetY );
	return f;
end

--[[
	
	args.menutable = {
		-- table of tables
		-- 
		{
			level 		= [INTEGER]; -- The menu level, 1 being bottom
			text 		= [STRING, function()]; -- (level == 1) The string or function returning a string to be displayed at the button. If a function, this will update whenever the user selects something
			value 		= [nil, STRING]; -- if this is a submenu, (level > 1) then this is the value (UIDROPDOWNMENU_MENU_VALUE) it is connected to
			menulist 	= {
				-- table of tables with the following options available;
				-- (cut and pasted from FrameXML\UIDropDownMenu.lua)
				{
					text 			= [STRING] -- The text of the button
					value 			= [ANYTHING] -- The value that UIDROPDOWNMENU_MENU_VALUE is set to when the button is clicked
					func 			= [function()] -- The function that is called when you click the button
					checked 			= [nil, true, function] -- Check the button if true or function returns true
					isTitle 			= [nil, true] -- If it's a title the button is disabled and the font color is set to yellow
					disabled 		= [nil, true] -- Disable the button and show an invisible button that still traps the mouseover event so menu doesn't time out
					hasArrow 		= [nil, true] -- Show the expand arrow for multilevel menus
					hasColorSwatch 	= [nil, true] -- Show color swatch or not, for color selection
					r 				= [1 - 255] -- Red color value of the color swatch
					g 				= [1 - 255] -- Green color value of the color swatch
					b 				= [1 - 255] -- Blue color value of the color swatch
					colorCode 		= [STRING] -- "|cAARRGGBB" embedded hex value of the button text color. Only used when button is enabled
					swatchFunc 		= [function()] -- Function called by the color picker on color change
					hasOpacity 		= [nil, 1] -- Show the opacity slider on the colorpicker frame
					opacity 			= [0.0 - 1.0] -- Percentatge of the opacity, 1.0 is fully shown, 0 is transparent
					opacityFunc 		= [function()] -- Function called by the opacity slider when you change its value
					cancelFunc 		= [function(previousValues)] -- Function called by the colorpicker when you click the cancel button (it takes the previous values as its argument)
					notClickable 		= [nil, 1] -- Disable the button and color the font white
					notCheckable 		= [nil, 1] -- Shrink the size of the buttons and don't display a check box
					owner 			= [Frame] -- Dropdown frame that "owns" the current dropdownlist
					keepShownOnClick 	= [nil, 1] -- Don't hide the dropdownlist after a button is clicked
					tooltipTitle 		= [nil, STRING] -- Title of the tooltip shown on mouseover
					tooltipText 		= [nil, STRING] -- Text of the tooltip shown on mouseover
					justifyH 		= [nil, "CENTER"] -- Justify button text
					arg1 			= [ANYTHING] -- This is the first argument used by func
					arg2 			= [ANYTHING] -- This is the second argument used by func
					fontObject 		= [FONT] -- font object replacement for Normal and Highlight
					menuTable 		= [TABLE] -- This contains an array of info tables to be displayed as a child menu
				};
			};
		};
	};
]]--
function lib:CreateDropDown( args )
	args = self:ArgumentCheck( args, {
		parent						= UIParent; 
		name 						= "";
		text 						= "";

		maxwidth 					= InterfaceOptionsFramePanelContainer:GetWidth() - 40;
		width 						= 100;
		padding 					= 2;
		justify 					= "LEFT";

		point 						= "TOPLEFT";
		anchor 						= UIParent; 
		relpoint 					= "TOPLEFT";
		offsetX 					= 0;
		offsetY 					= 0;

		menutable 					= {};
	});
	
	-- create our main button frame
	local button = CreateFrame("Frame", args.name, args.parent);
	button:SetPoint( args.relpoint, args.anchor, args.relpoint, args.offsetX, args.offsetY );
	button:SetWidth(args.maxwidth);
	button:SetHeight((select(2, GameFontNormal:GetFont())));
	
	-- create the description text for our dropdown
	button.text = self:CreateText({
		parent 				= button; 
		name 				= args.name.."Text";
		framestrata 		= "ARTWORK";
		font 				= (select(1, GameFontNormal:GetFont())); 
		fontsize 			= (select(2, GameFontNormal:GetFont())); 
		point 				= "TOPLEFT";
		relpoint 			= "TOPLEFT";
		anchor 				= button;
		justifyH 			= "LEFT";
		text 				= NORMAL_FONT_COLOR_CODE..args.text.."|r";
		maxwidth 			= args.maxwidth - args.width - 40;
	});
	
	-- create the dropdown frame
	button.dropdown = CreateFrame("Frame", args.name.."DropDown", args.parent, "UIDropDownMenuTemplate");
	button.dropdown:SetPoint( "TOPLEFT", button.text, "TOPRIGHT", 0, 8 );

	-- create a shortcut, since we're lazy
	local frame = button.dropdown;

	-- the table we store our buttons in
	local dd = {};
	
	-- our main loop to iterate over potential submenus
	for i,v in pairs(args.menutable) do
		local text 		= args.menutable[i].text;
		local value 	= args.menutable[i].value;
		local level 	= args.menutable[i].level or 1; -- if no level is given, we assume this is a single level menu
		local menulist 	= args.menutable[i].menulist;
		
		local pointer; 
		
		UIDropDownMenu_SetWidth(frame, args.width, args.padding);
		UIDropDownMenu_JustifyText(frame, args.justify);

		-- the text displayed on the button
		if level == 1 then 
			if type(text) == "function" then
				frame:HookScript("OnUpdate", function() UIDropDownMenu_SetText(frame, text()) end);
			elseif type(text) == "string" then
				UIDropDownMenu_SetText(frame, text);
			end
		end
		
		-- decide where in the dd table this info goes
		if level == 1 then
			-- create the top level
			-- we always assume this does not exist
			dd[level] 			= {};
			pointer 			= dd[level];
		else
			-- create the sublevel if it doesn't already exist
			dd[level] 			= dd[level] or {};
			dd[level][value] 	= {};
			pointer 			= dd[level][value];
		end
		
		-- create the menu items for this level and branch
		for j,w in pairs(args.menutable[i].menulist) do
			pointer[j] = {};
			for x,y in pairs(args.menutable[i].menulist[j]) do
				-- insert the values into the correct place in the dd table 
				pointer[j][x] = args.menutable[i].menulist[j][x];
			end
		end

	end
	

	local info = {}
	frame.initialize = function(self, level)
		if not level then return end
		wipe(info);
		local pointer 
		
		--  figure out which part of the dd table to get our info from
		if level == 1 then
			pointer = dd[level];
		elseif level > 1 then
			pointer = dd[level][UIDROPDOWNMENU_MENU_VALUE];
		end

		-- Create the buttons for this level and branch
		for i,v in pairs(pointer) do
			for j,w in pairs(v) do
				info[j] = w;
			end
			
			-- add the button
			UIDropDownMenu_AddButton(info, level);
		end
	end

	frame.Title = CreateFrame("Button", args.name.."_Button", frame);
	frame.Title:SetAllPoints(frame);
	frame.Title:SetScript("OnClick", function(self, button, down)
		if button == "RightButton" then
			ToggleDropDownMenu(1, nil, frame, self:GetName(), 0, 0, "MENU");
		end
	end)
	frame.Title:RegisterForClicks("RightButtonUp");

	local bh = max(frame.Title:GetHeight(), button.text:GetHeight(), button:GetHeight(), frame:GetHeight() );
	
	button:SetHeight(bh);
	
	return button
end

function lib:CreateSlider( args )
end

function lib:CreateColorPicker( args )
end




------------------------------------------------------------------------------------------------------------
--
-- gLib 2.x backwards compability
--
------------------------------------------------------------------------------------------------------------
function lib:Coin(money) return Tags.money(money) end


------------------------------------------------------------------------------------------------------------
--
--	gLib 1.x backward compability
-- 	Will remain here until all gMods are updated to the new functionality of gLib2
--
------------------------------------------------------------------------------------------------------------
function gLib:CreateOptionsPanel(panel)
	--[[
		Converts the array given in {panel} to an Interface Options menu,
		and returns the link to the new menu object.
		
		All the .name values are valid as links for the InterfaceOptionsFrame_OpenToCategory()
		function to open to.
		
		Format:

			panel.title = title of the addon to which the menu belongs
			panel.version = version number of the addon
			panel.description = description of the addon, the text shown on the menu's front page
			
			panel.option[x] = frontpage menu options array, se below for more settings
				panel.option[x].name = text on the menu option
				panel.option[x].type = what kind of menu option?
												"button" = normal checkbutton
				panel.option[x].value = value of the menu option
				panel.option[x].panelfunction = function connected to the menu option
			
			panel.menu[x] = submenu
				panel.menu[x].name = title of the subtree
				panel.menu[x].option[x] = array of menu options
				panel.menu[x].option[x].name = text on the menu option
				panel.menu[x].option[x].type = what kind of menu option?
												"button" = normal checkbutton
				panel.menu[x].option[x].value = value of the menu option
				panel.menu[x].option[x].panelfunction = function connected to the menu option
		
		Example:
			your_panel = gLib:CreateOptionsPanel(your_array)
			InterfaceOptionsFrame_OpenToCategory(your_array.menu[3].name)
			
			* 	Creates an Interface Options menu tree based on {your_array}, 
				and opens the menu at the 4th submenu.
	]]--

	-- create main container frame for the menu tree
	local p = CreateFrame("Frame","",UIParent);
	p.head = self:CreateText(p,panel.title.."OPTIONS_TITLE","ARTWORK","GameFontNormalLarge",_,_,p,14,-14,_,panel.title.." "..panel.version);
	p.name = panel.title.." "..panel.version;
	if panel.description then 
		p.body = self:CreateText(p,panel.title.."OPTIONS_DESCRIPTION", "ARTWORK", "GameFontWhite",_,_,p.head,8,-20,_,panel.description);
	end
	
	-- if it exists, show the frontpage's menu options
	if panel.option then
		p.option = {};
		local previousparent = p.head;
		for maincounter, c in pairs(panel.option) do
			local xoff,yoff = 0,-28; if maincounter == 0 and p.body then yoff = yoff - p.body:GetHeight(); end;
			if panel.option[maincounter].type == "button" then
				p.option[maincounter] = self:CreateCheckButton(p.name.."_CB"..maincounter, panel.option[maincounter].name, p,_,previousparent, xoff, yoff,panel.option[maincounter].value,_,_,panel.option[maincounter].panelfunction);
			elseif panel.option[maincounter].type == "submitbutton" then
				p.option[maincounter] = self:CreateButton(p.name.."_CB"..maincounter, panel.option[maincounter].name, p,"TOP",previousparent, xoff, yoff,panel.option[maincounter].panelfunction);
			end
			previousparent = p.option[maincounter];
		end
		p.option.parent = p.name;
	end

	p.okay = panel.okay or function () end;
	p.cancel = panel.cancel or function () end;
	p.default = panel.default or function () end;
	p.refresh = panel.refresh or function () end
	
	-- add main menu object
	InterfaceOptions_AddCategory(p);
	
	-- create submenus
	if panel.menu then
		p.menu = {};
		for counter, a in pairs(panel.menu) do
			p.menu[counter] = CreateFrame("Frame",panel.menu[counter].name,p);
			p.menu[counter].name = panel.menu[counter].name; 
			p.menu[counter].head = self:CreateText(p.menu[counter],p.menu[counter].name.."_TITLE", "ARTWORK", "GameFontNormalLarge",_,_,p.menu[counter], 14, -14,_,p.name.." "..panel.menu[counter].name);
			if panel.menu[counter].description then 
				p.menu[counter].body = self:CreateText(p.menu[counter],panel.menu[counter].name.."SUBOPTIONS_DESCRIPTION"..counter, "ARTWORK", "GameFontWhite",_,_,p.menu[counter].head,8,-20,_,panel.menu[counter].description);
			end
			p.menu[counter].option = {};
			local previousparent = p.menu[counter].head;
			
			for subcounter, b in pairs(panel.menu[counter].option) do
				local xoff,yoff = 0,-28;
				if subcounter == 0 and p.menu[counter].body then yoff = yoff - p.menu[counter].body:GetHeight(); end				
				if panel.menu[counter].option[subcounter].type == "button" then
					p.menu[counter].option[subcounter] = self:CreateCheckButton(p.menu[counter].name.."_CB"..subcounter, panel.menu[counter].option[subcounter].name, p.menu[counter],_,previousparent, xoff, yoff,panel.menu[counter].option[subcounter].value,_,_,panel.menu[counter].option[subcounter].panelfunction);
				end
				previousparent = p.menu[counter].option[subcounter];
			end
			p.menu[counter].parent = p.name;
			InterfaceOptions_AddCategory(p.menu[counter]);
		end
	end

	-- return the menu object
	return p
end
function gLib:CreateCheckButton(name, text, parent, relativepoint, anchorframe, offsetX, offsetY, checked, justifyH, justifyV, onclickscript)
	if name == nil then name = ""; end
	if parent == nil or parent == "" then parent = UIParent; end
	if anchorframe == nil or anchorframe == "" then anchorframe = UIParent; end
	if not(self.VALIDPOINTS[relativepoint]) then relativepoint = "TOPLEFT"; end
	if not((justifyH == "RIGHT") or (justifyH == "LEFT")) then justifyH = "LEFT"; end
	if not((justifyV == "TOP") or (justifyV == "BOTTOM")) then justifyV = "TOP"; end
	if onclickscript == nil or onclickscript == "" then onclickscript = function() end end

	local b = CreateFrame("CheckButton", name, parent, "InterfaceOptionsCheckButtonTemplate");
		b:SetPoint(relativepoint,anchorframe,relativepoint,offsetX,offsetY);
		b:SetChecked(checked);
		b:SetScript("OnClick",onclickscript);

		b.text = self:CreateText(b,name.."Text", "ARTWORK", "GameFontWhite","LEFT","RIGHT",b,_,_,_,text);
	return b;
end
function gLib:CreateButton(name, text, parent, relativepoint, anchorframe, offsetX, offsetY, onclickscript)
	if name == nil then name = ""; end
	if parent == nil or parent == "" then parent = UIParent; end
	if anchorframe == nil or anchorframe == "" then anchorframe = UIParent; end
	if not(self.VALIDPOINTS[relativepoint]) then relativepoint = "TOPLEFT"; end
	if onclickscript == nil or onclickscript == "" then onclickscript = function() end end

	local b = CreateFrame("Button", name, parent, "UIPanelButtonTemplate2");
		b:SetWidth(b:GetWidth()+40);
		b:SetPoint(relativepoint,anchorframe,relativepoint,offsetX,offsetY);
		b:SetScript("OnClick",onclickscript);
		b.text = self:CreateText(b,name.."Text", "ARTWORK", "GameFontWhite","LEFT","RIGHT",b,_,_,_,text);
	return b;
end
function gLib:CreateText(frame, name, framestrata, font, relativepoint, relativepointparent, anchorframe, offsetX, offsetY, justifyH, text)
	if name == nil then name = ""; end
	if parent == nil or parent == "" then parent = UIParent; end
	if anchorframe == nil or anchorframe == "" then anchorframe = UIParent; end
	if not(self.VALIDPOINTS[relativepoint]) then relativepoint = "TOPLEFT"; end
	if not(self.VALIDPOINTS[relativepointparent]) then relativepointparent = "TOPLEFT"; end
	if offsetX == nil or offsetX == "" then offsetX = 0; end
	if offsetY == nil or offsetY == "" then offsetY = 0; end
	if not((justifyH == "RIGHT") or (justifyH == "LEFT")) then justifyH = "LEFT"; end
	if not((justifyV == "TOP") or (justifyV == "BOTTOM")) then justifyV = "TOP"; end
	if text == nil then text = ""; end

	local f = frame:CreateFontString(name, framestrata, font);
		f:SetText(text);
		f:SetJustifyH(justifyH);
		f:SetPoint(relativepoint,anchorframe,relativepointparent,offsetX,offsetY);
	return f;
end
function gLib:print(header, ...)
	-- set output header
	header = header or ""
	
	--[[
		Prints out any number of system messages 
		on separated lines in the main chat frame.
		
		{...} can be any number of comma-separated messages, 
		or a table containing strings. 
		
		Empty values "" will add a blank line with no header message.
	]]--
	
	local args = {...};
	local function printmessage(args)
		for i, a in pairs(args) do
			-- call itself if the input string is a table
			if type(a) == "table" then 
				printmessage(a) 
			else
				DEFAULT_CHAT_FRAME:AddMessage(((a ~= "" ) and header or " ")..a);
			end
		end
	end
	printmessage(args);
end
function gLib:clone( base_object, clone_object )
	-- clones base_object and returns it as clone_object which can be optionally provided in the function call
	if base_object == nil then
		return nil
	end
	if type( base_object ) ~= "table" then
		return clone_object or base_object 
	end
	clone_object = clone_object or {}
	clone_object.__index = base_object
	return setmetatable(clone_object, clone_object)
end
function gLib:toggle(object,state)
	-- Hides/Shows any given element
	if state then
		object:Show();
	elseif not state then
		object:Hide();
	end
end
function gLib:GetPos(frame)
	local point, anchor, relpoint, x, y = frame:GetPoint();
	return { 
		point = point;
		anchor = type(anchor) == "string" and anchor or "UIParent"; 
		relpoint = relpoint; 
		x = tonumber(x);
		y = tonumber(y);
	}
end
function gLib:SetPos(frame, where)
	where = where or {}

	local point 	= self.VALIDPOINTS[where.point] and tostring(where.point) or "TOPLEFT"
	local anchor 	= tostring(where.anchor) or "UIParent"
	local relpoint 	= self.VALIDPOINTS[where.relpoint] and tostring(where.relpoint) or "TOPLEFT"
	local x 		= tonumber(where.x) or 0
	local y 		= tonumber(where.x) or 0

	frame:SetPoint(point, anchor, relpoint, x, y);
end