--[[
Name: oUF_DruidHots
Description: Provides Druid HoT indicators across the top of the unit frame

To disable the indicators for a given frame, set ignoreDruidHots
to any non-nil value, e.g.:

frame.ignoreDruidHots = true

Copyright 2008 Quaiche

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
]]

local parent = debugstack():match[[\AddOns\(.-)\]]
local global = GetAddOnMetadata(parent, 'X-oUF')
local oUF = _G[global] or oUF
assert(oUF, 'oUF not loaded')

local UnitBuff = _G.UnitBuff
local GetTime = _G.GetTime

assert(oUF, 'oUF not loaded')

if select(2,UnitClass("player")) ~= "DRUID" then return end

local debugf = tekDebug and tekDebug:GetFrame("oUF_DruidHots")
local function Debug(...) if debugf then debugf:AddMessage(string.join(", ", tostringall(...))) end end

-- Icon textures for local-neutral identification of buffs
local LIFEBLOOM = "Interface\\Icons\\INV_Misc_Herb_Felblossom"
local REGROWTH = "Interface\\Icons\\Spell_Nature_ResistNature"
local REJUVENATION = "Interface\\Icons\\Spell_Nature_Rejuvenation"
local WILDGROWTH = "Interface\\Icons\\Ability_Druid_Flourish"

-- Configuration
local width, height = 4, 3
local offsetX, offsetY = 5, -2
local spacing = 2
local order = { LIFEBLOOM, REJUVENATION, REGROWTH, WILDGROWTH }

local hots = {
	[LIFEBLOOM] = {
		count = 3,
		color = { r=0, g=1.0, b=0 },
	},
	[REGROWTH] = {
		count = 1,
		color = { r=0, g=0, b=1 },
	},
	[REJUVENATION] = {
		count = 1,
		color = { r=1, g=1, b=0 },
	},
	[WILDGROWTH] = {
		count = 1,
		color = { r=1, g=0.5, b=0 },
	},
}

local function SetStatus(object, texture, count, remaining)
	for i=1,hots[texture].count do
		if hots[texture].count > 1 and count >= i then
			object.druidHotsIndicators[texture][i]:Show()
		else
			object.druidHotsIndicators[texture][i]:Hide()
		end
	end
end

local state = {} -- Temp table to avoid memory thrashing
local function Update(object, event, unit)
	if object.unit ~= unit then return end

	local i = 1
	state[LIFEBLOOM] = 0
	state[REGROWTH] = 0
	state[REJUVENATION] = 0
	state[WILDGROWTH] = 0

	while true do
		local name, _, texture, count, _, _, expirationTime, caster = UnitBuff(unit, i, true)
		local isMine = (caster == "player")
		if not name then 
			break 
		end
		if isMine and hots[texture] then
			local timeleft = expirationTime - GetTime()
			if texture == LIFEBLOOM then
				state[texture] = count
			else
				state[texture] = 1
			end
		end
		i = i + 1
	end

	for k,v in pairs(state) do
		for i=1,hots[k].count do
			if i <= state[k] then
				object.druidHotsIndicators[k][i]:Show()
			else
				object.druidHotsIndicators[k][i]:Hide()
			end
		end
	end
end

local function initialize(object)
	if object.ignoreDruidHots or object.druidHotsIndicators then return end

	local i,j,k
	local tex
	local anchor, anchorPoint, x, y = object.Health, "TOPLEFT", offsetX, offsetY

	object.druidHotsIndicators = {}
	for j = 1,#order do
		k = order[j]
		object.druidHotsIndicators[k] = {}
		for i = 1,hots[k].count do
			tex = object.Health:CreateTexture(nil, "OVERLAY")
			tex:SetPoint("TOPLEFT", anchor, anchorPoint, x, y)
			tex:SetTexture(hots[k].color.r, hots[k].color.g, hots[k].color.b)
			tex:SetHeight(height)
			tex:SetWidth(width)
			tex:Hide()

			anchor, anchorPoint, x, y = tex, "TOPRIGHT", spacing, 0
			table.insert(object.druidHotsIndicators[k],tex)
		end
	end

	object:RegisterEvent("UNIT_AURA", Update)
end

-- Initialize the extension for existing frames and register for subsequent frames
for k, object in ipairs(oUF.objects) do initialize(object) end
oUF:RegisterInitCallback(initialize)

