--[[
	gFrames
	Author: Lars Norberg
	
	Supported units:
		Player
		Player Pet
		Target
		Target Target
		Focus
		Focus Target
		Party
		Party Pets
		Vehicles
	
	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is its designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat		
]]--

gFrames = CreateFrame("Frame", nil, UIParent);
gFrames.ADDON = "gFrames"; 
gFrames.VERSION = GetAddOnMetadata(gFrames.ADDON,"Version"); 
gFrames._G = _G

gFrames:SetScript("OnEvent", function(self, event, ...) self[event](self, ...) end)
gFrames:RegisterEvent("ADDON_LOADED")
gFrames:RegisterEvent("PLAYER_LOGIN")
gFrames:RegisterEvent("PLAYER_LEAVING_WORLD")

function gFrames:PLAYER_LEAVING_WORLD()
	-- save settings
	gFrames_userconfig.UNITFRAMES_OFFSET = gFrames_userconfig.UNITFRAMES_OFFSET or UNITFRAMES_OFFSET
	gFrames_userconfig.UI_LAYOUT = gFrames_userconfig.UI_LAYOUT or UI_LAYOUT
end
function gFrames:ADDON_LOADED()
	if not (arg1 == self.ADDON) then return end
	self:UnregisterEvent("ADDON_LOADED")

	-- retrieve gUI settings or saved settings
	gFrames_userconfig = gFrames_userconfig or {}
	gFrames_userconfig.UI_LAYOUT = tonumber((UI_LAYOUT or gFrames_userconfig.UI_LAYOUT) or 5)
	gFrames_userconfig.UNITFRAMES_OFFSET = tonumber((UNITFRAMES_OFFSET or gFrames_userconfig.UNITFRAMES_OFFSET) or 100)
end
function gFrames:PLAYER_LOGIN()
	self:UnregisterEvent("PLAYER_LOGIN")

	------------------------------------------------------------
	--		Variables
	------------------------------------------------------------

	local db = gFrames_db
	local coords, sizes, spawn = db.coords, db.sizes, db.spawn

	-- overlay colors used by the unitframes
	local overlaycolor, darkoverlaycolor = db.overlaycolor, db.darkoverlaycolor

	-- low and high mana threshold
	local lowThreshold, highThreshold = db.lowThreshold, db.highThreshold
	
	-- globals used by gUI
	local UI_LAYOUT = tonumber(UI_LAYOUT or gFrames_userconfig.UI_LAYOUT); 	if UI_LAYOUT == 6 then UI_LAYOUT = 3 end
	local PIXEL = (768 / tonumber(string.match(GetCVar("gxResolution"), "%d+x(%d+)"))) / tonumber(GetCVar("uiScale"))
	local PADDING = (PADDING or 2)*PIXEL
	local UIPADDING = (UIPADDING or 8)*PIXEL
	local UI_SHADELEVEL = 1
	local UNITFRAMES_OFFSET = tonumber(UNITFRAMES_OFFSET or gFrames_userconfig.UNITFRAMES_OFFSET)
	
	-- slash commands to toggle layouts and vertical offset
	if not(IsAddOnLoaded("gUI")) then
		self._G["SlashCmdList"]["gFrames_LAYOUT"] = function(layout) 
			local layoutconvertor = { [1] = 0, [2] = 5, [3] = 2 }
			if not(layoutconvertor[tonumber(layout)]) then 
				print("Valid UnitFrame layouts are 1-3");
			end
			gFrames_userconfig.UI_LAYOUT = layoutconvertor[tonumber(layout)]; 
			ReloadUI();
		end
		self._G["SLASH_gFrames_LAYOUT1"] = "/ulay"
	end
	self._G["SlashCmdList"]["gFrames_UNITFRAMES_OFFSET"] = function(offset) gFrames_userconfig.UNITFRAMES_OFFSET = tonumber(offset); ReloadUI(); end
	self._G["SLASH_gFrames_UNITFRAMES_OFFSET1"] = "/offset"
	
	-- shortcuts
	local floor 			= math.floor
	local format 			= string.format
	local gsub 				= string.gsub
	local lib 				= gMedia.lib
	
	-- backdrops
	local borderpadding 	= 2 * PIXEL
	local backdrop 			= lib.backdrop["default"].backdrop
	local blankbackdrop 	= lib.backdrop["blank"].backdrop
	local borderbackdrop 	= lib.backdrop["satin"].backdrop
	local glowbackdrop 		= lib.backdrop["glow"].backdrop
	
	
	-- textures
	local texture = {
		blank 				= lib.background["blank"];
		satin 				= lib.background["satin"];
		steel 				= lib.background["steel"];
		ace 				= lib.statusbar["acebar"];
		runes 				= lib.statusbar["acebar"]; 
		normal 				= lib.statusbar["bantobar"]; 
		bars 				= lib.statusbar["diagonal"];
		glow 				= lib.border["glow"];
		bubble 				= lib.icon["bubble"]; 
		button 				= lib.button["button"];
		highlight 			= lib.statusbar["highlight"]; 
		unitshader 			= lib.background["unitshader"];
	}

	-- fonts
	local fontname 			= lib.font["zrnic"]
	local bigfont 			= lib.font["zekton"]
	local numberfont 		= lib.font["bignoodletitling"]
	local fontheight 		= db.fontheight
	local fontmultiplier 	= db.fontmultiplier
	for i,v in pairs(fontheight) do
		fontheight[i] 		= fontheight[i] *PIXEL
	end

	-- get the player's class
	local class = (select(2,UnitClass("player")))

	-- our colors
	local colors = setmetatable({
		class = setmetatable({
			["HUNTER"] = 			{ 0.67 * 0.85, 0.83 * 0.85, 0.45 * 0.85 },
			["WARLOCK"] = 			{ 0.58 * 0.85, 0.51 * 0.85, 0.79 * 0.85 },
			["PRIEST"] = 			{ 0.85 * 0.85, 0.85 * 0.85, 0.85 * 0.85 },
			["PALADIN"] = 			{ 0.96 * 0.85, 0.55 * 0.85, 0.73 * 0.85 },
			["MAGE"] = 				{ 0.41 * 0.85, 0.80 * 0.85, 0.94 * 0.85 },
			["ROGUE"] = 			{ 1.00 * 0.85, 0.96 * 0.85, 0.41 * 0.85 },
			["DRUID"] = 			{ 1.00 * 0.85, 0.49 * 0.85, 0.04 * 0.85 },
			["SHAMAN"] = 			{ 0.00 * 0.85, 0.44 * 0.85, 0.87 * 0.85 },
			["WARRIOR"] = 			{ 0.78 * 0.85, 0.61 * 0.85, 0.43 * 0.85 },
			["DEATHKNIGHT"] = 		{ 0.77 * 0.85, 0.12 * 0.85, 0.23 * 0.85 },
		}, {__index = oUF.colors.class}),
		health = setmetatable({ 
			0.33, 0.59, 0.33
		}, {__index = oUF.colors.health}),
		power = setmetatable({
			["MANA"] = 			{ 0.15 * 0.85, 0.15 * 0.85, 0.85 * 0.85 };
			["RAGE"] = 			{ 1.00 * 0.85, 0.00 * 0.85, 0.00 * 0.85 };
			["FOCUS"] = 		{ 1.00 * 0.85, 0.50 * 0.85, 0.25 * 0.85 };
			["ENERGY"] = 		{ 1.00 * 0.85, 1.00 * 0.85, 0.00 * 0.85 };
			["HAPPINESS"] = 	{ 0.00 * 0.85, 1.00 * 0.85, 1.00 * 0.85 };
			["RUNES"] = 		{ 0.50 * 0.85, 0.50 * 0.85, 0.50 * 0.85 };
			["RUNIC_POWER"] = 	{ 0.00 * 0.85, 0.82 * 0.85, 1.00 * 0.85 };
			["AMMOSLOT"] = 		{ 0.80 * 0.85, 0.60 * 0.85, 0.00 * 0.85 };
			["FUEL"] = 			{ 0.00 * 0.85, 0.55 * 0.85, 0.50 * 0.85 };
		}, {__index = oUF.colors.power}),
		happiness = setmetatable({
			[1] = { 0.79, 0.25, 0.25},
			[2] = { 0.89, 0.69, 0.25},
			[3] = { RAID_CLASS_COLORS["HUNTER"].r * 0.66, RAID_CLASS_COLORS["HUNTER"].g * 0.66, RAID_CLASS_COLORS["HUNTER"].b * 0.66},
		}, {__index = oUF.colors.happiness}),
		smooth = setmetatable({
			0.79, 0.15, 0.15, 
			0.49, 0.15, 0.15, 
			0.15, 0.15, 0.15, 
		}, {__index = oUF.colors.smooth}),
		runes = setmetatable({
			[1]	= {0.7, 0.2, 0.2},	-- blood
			[2]	= {0.2, 0.5, 0.2},	-- unholy
			[3]	= {0.2, 0.6, 0.6},	-- frost
			[4]	= {0.4, 0.2, 0.6},	-- death
		}, {__index = oUF.colors.runes}),
		disconnected = setmetatable({ 
			0.6, 0.6, 0.6
			}, {__index = oUF.colors.tapped}),
		tapped = setmetatable({ 
			0.6, 0.6, 0.6 
		}, {__index = oUF.colors.tapped}),
	}, {__index = oUF.colors})

	
	------------------------------------------------------------
	--		Local Functions
	------------------------------------------------------------

	local SetUpAnimGroup = function(self)
		self.anim = self:CreateAnimationGroup("Flash")
		self.anim.fadein = self.anim:CreateAnimation("ALPHA", "FadeIn")
		self.anim.fadein:SetChange(1)
		self.anim.fadein:SetOrder(2)

		self.anim.fadeout = self.anim:CreateAnimation("ALPHA", "FadeOut")
		self.anim.fadeout:SetChange(-1)
		self.anim.fadeout:SetOrder(1)
	end

	local Flash = function(self, duration)
		if not self.anim then
			SetUpAnimGroup(self)
		end

		self.anim.fadein:SetDuration(duration)
		self.anim.fadeout:SetDuration(duration)
		self.anim:Play()
	end

	local StopFlash = function(self)
		if self.anim then
			self.anim:Finish()
		end
	end

	local Menu = function(self)
		local unit = self.unit:gsub("(.)", string.upper, 1) 
		if _G[unit.."FrameDropDown"] then
			ToggleDropDownMenu(1, nil, _G[unit.."FrameDropDown"], "cursor")
		elseif (self.unit:match("party")) then
			ToggleDropDownMenu(1, nil, _G["PartyMemberFrame"..self.id.."DropDown"], "cursor")
		else
			FriendsDropDown.unit = self.unit
			FriendsDropDown.id = self.id
			FriendsDropDown.initialize = RaidFrameDropDown_Initialize
			ToggleDropDownMenu(1, nil, FriendsDropDown, "cursor")
		end
	end
		
	local PostUpdateHealth = function(self, event, unit, bar, min, max)
		if not UnitIsConnected(unit) then
			bar:SetValue(0)
			bar.value:SetText("|cffD7BEA5".."Off".."|r")
		elseif UnitIsDead(unit) then
			bar.value:SetText("|cffD7BEA5".."Dead".."|r")
		elseif UnitIsGhost(unit) then
			bar.value:SetText("|cffD7BEA5".."Ghost".."|r")
		else
			if min ~= max then
				if (unit == "player" and self:GetAttribute("normalUnit") ~= "pet") or (unit == "target") then
					bar.value:SetFormattedText("|cffffffff%s|r |cffD7BEA5-|r |cff%02x%02x%02x%d%%|r", gMedia:shortValue(min), 255, 255, 255, floor(min / max * 100))
				else
					bar.value:SetFormattedText("|cff%02x%02x%02x%d%%|r", 255, 255, 255, floor(min / max * 100))
				end
			else
				bar.value:SetText("|cffffffff"..gMedia:shortValue(max).."|r")
			end
		end
	end

	local PostUpdatePower = function(self, event, unit, bar, min, max)
		if self.unit ~= "player" and self.unit ~= "target" then return end

		local pType, pToken = UnitPowerType(unit)
		bar.value:SetTextColor(1, 1, 1)

		if min == 0 then
			bar.value:SetText()
		elseif not UnitIsPlayer(unit) and not UnitPlayerControlled(unit) or not UnitIsConnected(unit) then
			bar.value:SetText()
		elseif UnitIsDead(unit) or UnitIsGhost(unit) then
			bar.value:SetText()
		elseif min == max and (pType == 2 or pType == 3 and pToken ~= "POWER_TYPE_PYRITE") then
			bar.value:SetText()
		else
			if min ~= max then
				if pType == 0 then
					if unit == "target" or unit == "player" then
						bar.value:SetFormattedText("%d%% |cffD7BEA5-|r %s", floor(min / max * 100), gMedia:shortValue(max - (max - min)))
					elseif unit == "player" and self:GetAttribute("normalUnit") == "pet" or unit == "pet" then
						bar.value:SetFormattedText("%d%%", floor(min / max * 100))
					else
						bar.value:SetFormattedText("%d%% |cffD7BEA5-|r %d", floor(min / max * 100), max - (max - min))
					end
				else
					bar.value:SetText(max - (max - min))
				end
			else
				if unit == "pet" then
					if db.showpetmana then bar.value:SetText(gMedia:shortValue(min)) end
				elseif unit == "target" or unit == "player" then
					bar.value:SetText(gMedia:shortValue(min))
				else
					bar.value:SetText(min)
				end
			end
		end
	end

	do
		local f = CreateFrame("Frame")
		local entering

		f:RegisterEvent("UNIT_ENTERED_VEHICLE")
		f:RegisterEvent("UNIT_EXITED_VEHICLE")
		
		local delay = 0.5
		local OnUpdate = function(self, elapsed)
			self.elapsed = (self.elapsed or delay) - elapsed
			if self.elapsed <= 0 then
				local petframe = oUF_pet
				petframe:PLAYER_ENTERING_WORLD()
				self:SetScript("OnUpdate", nil)
				if entering and petframe.PostEnterVehicle then
					petframe:PostEnterVehicle("enter")
				elseif not entering and petframe.PostExitVehicle then
					petframe:PostExitVehicle("exit")
				end
			end
		end

		f:SetScript("OnEvent", function(self, event, unit)
			if unit == "player" then
				if event == "UNIT_ENTERED_VEHICLE" then
					entering = true
				else
					entering = false
				end
				f.elapsed = delay
				f:SetScript("OnUpdate", OnUpdate)
			end
		end)
	end

	local delay = 0
	local viperAspectName = GetSpellInfo(34074)
	local UpdateManaLevel = function(self, elapsed)
		delay = delay + elapsed
		if self.parent.unit ~= "player" or delay < 0.2 or UnitIsDeadOrGhost("player") or UnitPowerType("player") ~= 0 then return end
		delay = 0

		local percMana = UnitMana("player") / UnitManaMax("player") * 100

		if AotV then
			local viper = UnitBuff("player", viperAspectName)
			if percMana >= highThreshold and viper then
				self.ManaLevel:SetText("|cffaf5050GO HAWK|r")
				Flash(self, 0.3)
			elseif percMana <= lowThreshold and not viper then
				self.ManaLevel:SetText("|cffaf5050GO VIPER|r")
				Flash(self, 0.3)
			else
				self.ManaLevel:SetText()
				StopFlash(self)
			end
		else
			if percMana <= lowThreshold then
				self.ManaLevel:SetText("|cffaf5050LOW MANA|r")
				Flash(self, 0.3)
			else
				self.ManaLevel:SetText()
				StopFlash(self)
			end
		end
	end

	local UpdateDruidMana = function(self)
		if self.unit ~= "player" then return end

		local num, str = UnitPowerType("player")
		if num ~= 0 then
			local min = UnitPower("player", 0)
			local max = UnitPowerMax("player", 0)

			local percMana = min / max * 100
			if percMana <= lowThreshold then
				self.FlashInfo.ManaLevel:SetText("|cffaf5050LOW MANA|r")
				Flash(self.FlashInfo, 0.3)
			else
				self.FlashInfo.ManaLevel:SetText()
				StopFlash(self.FlashInfo)
			end

			if min ~= max then
				self.DruidMana:SetFormattedText("%d%%", floor(min / max * 100))
			else
				self.DruidMana:SetText()
			end

			self.DruidMana:SetAlpha(1)
		else
			self.DruidMana:SetAlpha(0)
		end
	end

	local UpdateCPoints = function(self, event, unit)
		if unit == PlayerFrame.unit and unit ~= self.CPoints.unit then
			self.CPoints.unit = unit
		end
	end

	local FormatCastbarTime = function(self, duration)
		if self.channeling then
			self.Time:SetFormattedText("%.1f ", duration)
		elseif self.casting then
			self.Time:SetFormattedText("%.1f ", self.max - duration)
		end
	end

	local CreateAuraTimer = function(self,elapsed)
		if self.timeLeft then
			self.elapsed = (self.elapsed or 0) + elapsed
			if self.elapsed >= 0.1 then
				if not self.first then
					self.timeLeft = self.timeLeft - self.elapsed
				else
					self.timeLeft = self.timeLeft - GetTime()
					self.first = false
				end
				if self.timeLeft > 0 then
					local time = gMedia:formatTime(self.timeLeft)
					if type(time) == "string" or time >= 10 then
						self.remaining:SetText(time)
					else
						self.remaining:SetFormattedText("%.1f", time)
					end
					self.remaining:SetTextColor(1, 1, 1)
				else
					self.remaining:Hide()
					self:SetScript("OnUpdate", nil)
				end
				self.elapsed = 0
			end
		end
	end

	local HideAura = function(self)
		if self.unit == "player" then
			if db.noPlayerAuras then
				self.Auras:Hide()
			else
				BuffFrame:Hide()
				TemporaryEnchantFrame:Hide()
			end
		elseif self.unit == "target" and db.noTargetAuras then
			self.Auras:Hide()
		end
	end

	local CancelAura = function(self, button)
		if button == "RightButton" and not self.debuff then
			CancelUnitBuff("player", self:GetID())
		end
	end

	local CreateAura = function(self, button, icons)
		button.count:SetPoint("BOTTOMRIGHT", -PIXEL, PIXEL)
		button.count:SetJustifyH("RIGHT")
		button.count:SetFont(numberfont, fontheight[8], "THICKOUTLINE")
		button.count:SetTextColor(1, 1, 1)

		button.cd.noOCC = true
		button.cd.noCooldownCount = true
		icons.disableCooldown = true

		button.overlay:SetTexture(texture.button)
		button.overlay:SetPoint("TOPLEFT", button, "TOPLEFT", -PIXEL, PIXEL)
		button.overlay:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", PIXEL, -PIXEL)
		button.overlay:SetTexCoord(0, 1, 0.02, 1)
		button.overlay.Hide = function(self) end

		if icons ~= self.Enchant then
			button.remaining = gMedia:makeFontString(button, numberfont, fontheight[10], "OUTLINE")
			if self.unit == "player" then
				button:SetScript("OnMouseUp", CancelAura)
			end
		else
			button.remaining = gMedia:makeFontString(button, numberfont, fontheight[10], "OUTLINE")
			button.overlay:SetVertexColor(0.33, 0.59, 0.33)
		end
		button.remaining:SetPoint("TOPLEFT", PIXEL, -PIXEL)
	end

	local CreateEnchantTimer = function(self, icons)
		for i = 1, 2 do
			local icon = icons[i]
			if icon.expTime then
				icon.timeLeft = icon.expTime - GetTime()
				icon.remaining:Show()
			else
				icon.remaining:Hide()
			end
			icon:SetScript("OnUpdate", CreateAuraTimer)
		end
	end

	local UpdateAura = function(self, icons, unit, icon, index)
		local _, _, _, _, _, duration, expirationTime, unitCaster, _ = UnitAura(unit, index, icon.filter)
		if unitCaster == "player" or unitCaster == "pet" or unitCaster == "vehicle" then
			if icon.debuff then
				icon.overlay:SetVertexColor(0.79, 0.15, 0.15)
			else
				icon.overlay:SetVertexColor(0.45, 0.45, 0.45)
			end
		else
			if UnitIsEnemy("player", unit) then
				if icon.debuff then
					icon.icon:SetDesaturated(true)
				end
			end
		end

		if duration and duration > 0 then
			icon.remaining:Show()
		else
			icon.remaining:Hide()
		end

		icon.duration = duration
		icon.timeLeft = expirationTime
		icon.first = true
		icon:SetScript("OnUpdate", CreateAuraTimer)
	end

	local HidePortrait = function(self, unit)
		if self.unit == "target" then
			if not UnitExists(self.unit) or not UnitIsConnected(self.unit) or not UnitIsVisible(self.unit) then
				self.Portrait:SetAlpha(0)
			else
				self.Portrait:SetAlpha(1)
			end
		end
	end
	
	local CropString = function( str, maxlength )
		if string.len(str) <= maxlength then return str end
		return string.sub(str, 1, maxlength-3 ).."..."
	end

	local OverrideUpdateThreat = function(self, event, unit, status)
		if (status and status > 0) and (GetNumRaidMembers() > 0) then
			local r, g, b = GetThreatStatusColor(status)
			self.Threat:SetBackdropBorderColor(r * 0.5, g * 0.5, b * 0.5, 1)
			self.Threat:Show()
		else
			self.Threat:SetBackdropBorderColor( unpack(gMedia.backdrops["SATIN"].bordercolor) )
			self.Threat:Show()
		end		
	end

	------------------------------------------------------------
	--		UnitFrame Styles
	------------------------------------------------------------
	local Styles = function(self, unit)
		self.menu = Menu
		self.colors = colors
		
		self:RegisterForClicks("AnyUp")
		self:SetAttribute("type2", "menu")

		self:SetScript("OnEnter", UnitFrame_OnEnter)
		self:SetScript("OnLeave", UnitFrame_OnLeave)
		
		self:SetFrameStrata("LOW")
		self:SetFrameLevel(5)
		
		------------------------------------------------------------
		--		Frame borders and shadows
		------------------------------------------------------------
		self.FrameBorder = gMedia:panel({parent = self, padding = borderpadding, overlay = 0, framestrata = "BACKGROUND" })
		self.FrameBorder:SetAllPoints(self)

		self.Threat = self.FrameBorder
		
		self.ignoreDruidHots = true
		self.BarFade = false

		-- Spellrange alpha
		self.SpellRange = db.spellrangefade;
		self.inRangeAlpha = 1.0; 
		self.outsideRangeAlpha = 0.3; 	
		
		------------------------------------------------------------
		--		Health
		------------------------------------------------------------
		self.Health = CreateFrame("StatusBar", self:GetName().."_Health", self)

		if unit then
			self.Health:SetFrameLevel(1)
		elseif self:GetAttribute("unitsuffix") then
			self.Health:SetFrameLevel(3)
		elseif not unit then
			self.Health:SetFrameLevel(2)
		end

		-- Decide whether we need extra space for a class specific bar or not (totems, runes)
		local classbar = (unit == "player" and ((class == "SHAMAN") or (class == "DEATHKNIGHT"))) and PIXEL*(sizes.pbheight[UI_LAYOUT].player + PIXEL) or 0;

		-- set the size of the healthbars for units other than the target and the player
		local pbsize = PIXEL*sizes.pbheight[UI_LAYOUT].default + PIXEL; 	-- height of the powerbars + 1
		self.Health:SetHeight((self:GetParent():GetName():match("oUF_Party") and PIXEL*sizes.partyH[UI_LAYOUT]-pbsize) or ((unit and unit:find("arena%d")) and PIXEL*sizes.arenaH[UI_LAYOUT]-pbsize) or (self:GetAttribute("unitsuffix") == "pet" and PIXEL*sizes.partyPetH[UI_LAYOUT]-pbsize) or (unit == "targettarget" and PIXEL*sizes.targettargetH[UI_LAYOUT]-pbsize) or (unit == "pet" and PIXEL*sizes.playerPetH[UI_LAYOUT]-pbsize) or (unit == "focus" and PIXEL*sizes.focusH[UI_LAYOUT]-pbsize) or (unit == "focustarget" and PIXEL*sizes.focustargetH[UI_LAYOUT]-pbsize) or PIXEL*sizes.defaultH[UI_LAYOUT]-pbsize)
		
		local healthTexture = texture.normal
		if (unit == "player") or (unit == "target") then
			self.Health:SetHeight(PIXEL*sizes.pbheight[UI_LAYOUT].player)
			self.Health:SetPoint("BOTTOMLEFT",self,"BOTTOMLEFT", borderpadding, PIXEL*sizes.pbheight[UI_LAYOUT].player +PIXEL + borderpadding)
			self.Health:SetPoint("BOTTOMRIGHT",self,"BOTTOMRIGHT", -borderpadding, PIXEL*sizes.pbheight[UI_LAYOUT].player +PIXEL + borderpadding)
		else
			healthTexture = texture.ace
			self.Health:SetPoint("TOPLEFT", self, "TOPLEFT", borderpadding, -borderpadding )
			self.Health:SetPoint("BOTTOMRIGHT", self, "BOTTOMRIGHT", -borderpadding, borderpadding + pbsize)
		end
		
		self.Health:SetStatusBarTexture(healthTexture)

		self.Health.colorTapping 			= db.HealthcolorTapping
		self.Health.colorDisconnected 		= db.HealthcolorDisconnected
		self.Health.colorSmooth 			= db.HealthcolorSmooth
		self.Health.colorClass 				= db.HealthcolorClass
		self.Health.colorClassNPC 			= db.HealthcolorClassNPC
		self.Health.colorHappiness 			= db.HealthcolorHappiness
		self.Health.colorHealth 			= true
		self.Health.frequentUpdates 		= true
		self.Health.Smooth 					= true

		self.Health.bg = self.Health:CreateTexture(nil, "BORDER")
		self.Health.bg:SetAllPoints(self.Health)
		self.Health.bg:SetTexture(healthTexture)
		self.Health.bg.multiplier = 0.33

		self.HealthOverlay = CreateFrame("StatusBar", self:GetName().."_HealthOverlay", self.Health)
		self.HealthOverlay:SetAllPoints(self.Health)
		self.HealthOverlay:SetFrameLevel(self.Health:GetFrameLevel() + 10)

		local hbfsize = fontmultiplier[UI_LAYOUT] * fontheight[14]
		
		if unit == "target" or unit == "player" then
			self.Health.value = gMedia:makeFontString(self.HealthOverlay, numberfont, hbfsize, "OUTLINE")
			self.Health.value:SetPoint("BOTTOMRIGHT", -PIXEL, 0)
		elseif self:GetParent():GetName():match("oUF_Party") or unit == "focus" or unit == "focustarget" then
			self.Health.value = gMedia:makeFontString(self.HealthOverlay, numberfont, hbfsize, "OUTLINE")
			self.Health.value:SetPoint("RIGHT", -PIXEL, 0)
		else
			self.Health.value = gMedia:makeFontString(self.HealthOverlay, numberfont, fontmultiplier[UI_LAYOUT] * 12, "OUTLINE")
			self.Health.value:SetPoint("RIGHT", -PIXEL, 0)
		end

		if unit ~= "player" and unit ~= "target" then
			if self:GetParent():GetName():match("oUF_Party") or unit == "focus" or unit == "focustarget" or ((unit == "pet" or unit == "targettarget") and (UI_LAYOUT == 0 or UI_LAYOUT == 1)) then
				self.BigName = self:CreateFontString(nil, "OVERLAY")
				self.BigName = gMedia:makeFontString(self.HealthOverlay, fontname, fontmultiplier[UI_LAYOUT] * fontheight[12], "OUTLINE")
				self.BigName:SetTextColor(1,1,1)
				self.BigName:SetPoint("LEFT", self.HealthOverlay, "LEFT", 4*PIXEL, 0)
				self:Tag(self.BigName, "[NameLong]")
			else
				self.BigName = self:CreateFontString(nil, "OVERLAY")
				self.BigName = gMedia:makeFontString(self.HealthOverlay, fontname, fontmultiplier[UI_LAYOUT] * fontheight[12], "OUTLINE")
				self.BigName:SetTextColor(1,1,1)
				self.BigName:SetPoint("LEFT", self.HealthOverlay, "LEFT", 1*PIXEL, 0)
				self:Tag(self.BigName, "[NameMedium]")
			end
		end

		------------------------------------------------------------
		--		Power
		------------------------------------------------------------
		if not (self:GetAttribute("unitsuffix") == "pet") then
			self.Power = CreateFrame("StatusBar", self:GetName().."_Power", self)
			self.Power:SetHeight((unit == "player" or unit == "target") and PIXEL * sizes.pbheight[UI_LAYOUT].player or PIXEL * sizes.pbheight[UI_LAYOUT].default)
			self.Power:SetPoint("BOTTOMLEFT", borderpadding, borderpadding)
			self.Power:SetPoint("BOTTOMRIGHT", -borderpadding, borderpadding)
			self.Power:SetStatusBarTexture(texture.normal)

			self.Power.colorTapping 			= db.PowercolorTapping
			self.Power.colorDisconnected 		= db.PowercolorDisconnected
			self.Power.colorClass 				= db.PowercolorClass
			self.Power.colorClassNPC 			= db.PowercolorClassNPC
			self.Power.colorReaction 			= db.PowercolorReaction
			self.Power.colorHappiness 			= db.PowercolorHappiness
			self.Power.colorPower 				= true
			self.Power.frequentUpdates 			= true
			self.Power.Smooth 					= true

			self.Power.upperline = CreateFrame("StatusBar", nil, self.Power)
			self.Power.upperline:SetStatusBarTexture(texture.blank)
			self.Power.upperline:SetStatusBarColor(0, 0, 0, 1)
			self.Power.upperline:SetHeight(PIXEL)
			self.Power.upperline:SetPoint("TOPLEFT", self.Power, "TOPLEFT", 0, PIXEL)
			self.Power.upperline:SetPoint("TOPRIGHT", self.Power, "TOPRIGHT", 0, PIXEL)
			
			self.Power.bg = self.Power:CreateTexture(nil, "BORDER")
			self.Power.bg:SetAllPoints(self.Power)
			self.Power.bg:SetTexture(texture.normal)
			self.Power.bg.multiplier = 0.33

			self.PowerOverlay = CreateFrame("StatusBar", self:GetName().."_Power", self.Power)
			self.PowerOverlay:SetAllPoints(self.Power)

			self.Power.value = gMedia:makeFontString(self.PowerOverlay, numberfont, fontmultiplier[UI_LAYOUT] * fontheight[12], "OUTLINE")
			self.Power.value:SetPoint("TOPRIGHT", self.PowerOverlay, "TOPRIGHT", -PIXEL, 0)
		end

		------------------------------------------------------------
		--		Death Knight Runebar
		------------------------------------------------------------
		if class == "DEATHKNIGHT" and unit == "player" then
			self.Runes = CreateFrame('Frame', nil, self)
			self.Runes:SetPoint('BOTTOMLEFT', self.Health, 'TOPLEFT', 0, PIXEL)
			self.Runes:SetHeight(PIXEL*sizes.pbheight[UI_LAYOUT].player + PIXEL)
			self.Runes:SetWidth(PIXEL*sizes.playerW[UI_LAYOUT] - borderpadding*2)
			self.Runes:SetFrameStrata("LOW")
			self.Runes:SetBackdrop(texture.blankbackdrop)
			self.Runes:SetBackdropColor(0.15, 0.15, 0.15, 1)
			self.Runes.anchor 	= "BOTTOMLEFT"
			self.Runes.growth 	= "RIGHT"
			self.Runes.height 	= PIXEL*sizes.pbheight[UI_LAYOUT].player
			self.Runes.width 	= (PIXEL*sizes.playerW[UI_LAYOUT] - borderpadding*2)/6 - PIXEL
			for i = 1, 6 do
				self.Runes[i] = CreateFrame("StatusBar", self:GetName().."_Runes"..i, self.Runes)

				self.Runes[i]:SetStatusBarTexture(texture.runes)
				self.Runes[i]:SetAlpha(1)
				self.Runes[i]:SetFrameStrata("LOW")
				self.Runes[i]:SetPoint(unpack(((i == 1) and {"TOPLEFT", self.Runes, "TOPLEFT", 0, 0}) or {"TOPLEFT", self.Runes[i-1], "TOPRIGHT", PIXEL, 0}))
				if i == 6 then self.Runes[i]:SetPoint("BOTTOMRIGHT", self.Runes, "BOTTOMRIGHT", 0, 0); end
				self.Runes[i].bg = self.Runes[i]:CreateTexture(nil, "BORDER")
				self.Runes[i].bg:SetTexture(texture.runes)
				self.Runes[i].bg:SetAllPoints(self.Runes[i])
				self.Runes[i].bg:SetVertexColor(0.15, 0.15, 0.15, 0.33)
			end
			self.Runes.bottomline = CreateFrame("StatusBar", nil, self.Runes)
			self.Runes.bottomline:SetStatusBarTexture(texture.blank)
			self.Runes.bottomline:SetStatusBarColor(0, 0, 0, 1)
			self.Runes.bottomline:SetHeight(PIXEL)
			self.Runes.bottomline:SetPoint("BOTTOMLEFT", self.Runes, "BOTTOMLEFT", 0, -PIXEL)
			self.Runes.bottomline:SetPoint("BOTTOMRIGHT", self.Runes, "BOTTOMRIGHT", 0, -PIXEL)
		end
		
		------------------------------------------------------------
		--		Shaman Totembar
		------------------------------------------------------------
		if class == "SHAMAN" and unit == "player" then
			self.TotemBar = CreateFrame('Frame', nil, self)
			self.TotemBar:SetPoint('BOTTOMLEFT', self.Health, 'TOPLEFT', 0, PIXEL)
			self.TotemBar:SetHeight(PIXEL*sizes.pbheight[UI_LAYOUT].player)
			self.TotemBar:SetWidth(PIXEL*sizes.playerW[UI_LAYOUT] - borderpadding*2)
			self.TotemBar:SetBackdrop(texture.blankbackdrop)
			self.TotemBar:SetBackdropColor(0, 0, 0)
			self.TotemBar.anchor 	= "BOTTOMLEFT"
			self.TotemBar.growth 	= "RIGHT"
			local lasttotemsize 	= ((PIXEL*sizes.playerW[UI_LAYOUT] - borderpadding*2) - floor((PIXEL*sizes.playerW[UI_LAYOUT] - borderpadding*2)/4)*3)
			local totemsize 		= floor((PIXEL*sizes.playerW[UI_LAYOUT] - borderpadding*2)/4)-PIXEL
			for i = 1, 4 do
				self.TotemBar[i] = CreateFrame("StatusBar", self:GetName().."_TotemBar"..i, self)
				self.TotemBar[i]:SetHeight(PIXEL*sizes.pbheight[UI_LAYOUT].player)
				self.TotemBar[i]:SetWidth((i == 4) and lasttotemsize or totemsize)
				if (i == 1) then
					self.TotemBar[i]:SetPoint("BOTTOMLEFT", self.TotemBar, "BOTTOMLEFT", 0, 0)
				else
					self.TotemBar[i]:SetPoint("TOPLEFT", self.TotemBar[i-1], "TOPRIGHT", PIXEL, 0)
				end
				self.TotemBar[i]:SetStatusBarTexture(texture.runes)
				self.TotemBar[i]:SetMinMaxValues(0, 1)
				self.TotemBar[i]:SetAlpha(1)
				self.TotemBar[i].bg = self.TotemBar[i]:CreateTexture(nil, "BORDER")
				self.TotemBar[i].bg:SetAllPoints(self.TotemBar[i])
				self.TotemBar[i].bg:SetTexture(texture.runes)
				self.TotemBar[i].bg:SetVertexColor(0.15, 0.15, 0.15, 1)
				self.TotemBar[i].bg.multiplier = 0.33
			end
			self.TotemBar.upperline = CreateFrame("StatusBar", nil, self.TotemBar)
			self.TotemBar.upperline:SetStatusBarTexture(texture.blank)
			self.TotemBar.upperline:SetStatusBarColor(0, 0, 0, 1)
			self.TotemBar.upperline:SetHeight(PIXEL)
			self.TotemBar.upperline:SetPoint("TOPLEFT", self.TotemBar, "TOPLEFT", 0, PIXEL)
			self.TotemBar.upperline:SetPoint("TOPRIGHT", self.TotemBar, "TOPRIGHT", 0, PIXEL)
		end

		------------------------------------------------------------
		--		Weapon Buffs
		------------------------------------------------------------
		if unit == "player" then
			self.Enchant = CreateFrame("Frame", nil, self)
			self.Enchant:SetHeight(22*PIXEL)
			self.Enchant:SetWidth(24 * 2 * PIXEL)
			self.Enchant:SetPoint("BOTTOMLEFT", self, "TOPLEFT", 0, 2*PIXEL)
			self.Enchant.size = 22*PIXEL
			self.Enchant.spacing = 2*PIXEL
			self.Enchant.showBlizzard = false
			self.Enchant.initialAnchor = "BOTTOMLEFT"
			self.Enchant["growth-x"] = "RIGHT"
			self.Enchant["growth-y"] = "UP"
		end

		------------------------------------------------------------
		--		Druid Mana in forms
		------------------------------------------------------------
		if class == "DRUID" and unit == "player" then
			CreateFrame("Frame"):SetScript("OnUpdate", function() UpdateDruidMana(self) end)
			self.DruidMana = gMedia:makeFontString(self.HealthOverlay, numberfont, fontmultiplier[UI_LAYOUT] * fontheight[12], "OUTLINE")
			self.DruidMana:SetPoint("BOTTOMLEFT", self.Health, "BOTTOMLEFT", PIXEL, 0)
			self.DruidMana:SetTextColor(1, 1, 1)
		end

		------------------------------------------------------------
		--		Player and Target auras
		------------------------------------------------------------
		if unit == "target" or unit == "player" then
			local unitbuffsize = 22*PIXEL
			local unitbuffspacing = 2*PIXEL
			local maxbuffs = floor(PIXEL*sizes.playerW[UI_LAYOUT] / (unitbuffsize))
			local unitbuffs = floor(maxbuffs/2)
			local unitdebuffs = maxbuffs - unitbuffs

			self.Auras = CreateFrame("Frame", nil, self)
			self.Auras:SetHeight(unitbuffsize)
			self.Auras:SetWidth((unitbuffsize + unitbuffspacing) * maxbuffs)
			self.Auras.size = unitbuffsize
			self.Auras.spacing = unitbuffspacing
			self.Auras.gap = false
			self.Auras:SetPoint("BOTTOMRIGHT", self, "TOPRIGHT", 0, 2*PIXEL)
			self.Auras.initialAnchor = "BOTTOMRIGHT"
			self.Auras["growth-y"] = "UP"
			self.Auras["growth-x"] = "LEFT"
			self.Auras.onlyShowPlayer = false

			if unit == "target" then
				self.CPoints = {}
				self.CPoints.unit = PlayerFrame.unit
				for i = 1, 5 do
					self.CPoints[i] = self.Power:CreateTexture(nil, "OVERLAY")
					self.CPoints[i]:SetHeight(12*PIXEL)
					self.CPoints[i]:SetWidth(12*PIXEL)
					self.CPoints[i]:SetTexture(texture.bubble)
					if i == 1 then
						self.CPoints[i]:SetPoint("LEFT")
						self.CPoints[i]:SetVertexColor(0.69, 0.31, 0.31)
					else
						self.CPoints[i]:SetPoint("LEFT", self.CPoints[i-1], "RIGHT", PIXEL)
					end
				end

				self.CPoints[2]:SetVertexColor(0.69, 0.31, 0.31)
				self.CPoints[3]:SetVertexColor(0.65, 0.63, 0.35)
				self.CPoints[4]:SetVertexColor(0.65, 0.63, 0.35)
				self.CPoints[5]:SetVertexColor(0.33, 0.59, 0.33)
				self:RegisterEvent("UNIT_COMBO_POINTS", UpdateCPoints)
			end
		end
		
		------------------------------------------------------------
		--		Portraits
		------------------------------------------------------------
		if unit == "player" or unit == "target" then
			self.Portrait = CreateFrame("PlayerModel", self:GetName().."_Portrait", self)
			self.Portrait:SetFrameLevel(1)
			self.Portrait:ClearAllPoints()
			self.Portrait:SetPoint("TOPLEFT", self, "TOPLEFT", (borderpadding + PIXEL), -(borderpadding + PIXEL))
			self.Portrait:SetPoint("BOTTOMRIGHT", self, "BOTTOMRIGHT", -(borderpadding + PIXEL), (borderpadding + 2 * (PIXEL*sizes.pbheight[UI_LAYOUT].player + PIXEL) + classbar))
			table.insert(self.__elements, HidePortrait)
			
			self.PortraitOverlay = CreateFrame("StatusBar", self:GetName().."_PortraitOverlay", self.Portrait)
			self.PortraitOverlay:SetBackdrop(blankbackdrop)
			self.PortraitOverlay:SetBackdropColor(0, 0, 0, 0.5)
			self.PortraitOverlay:SetFrameLevel(2)
			self.PortraitOverlay:SetPoint("TOPLEFT", self.Portrait, "TOPLEFT", -PIXEL, PIXEL)
			self.PortraitOverlay:SetPoint("BOTTOMRIGHT", self.Portrait, "BOTTOMRIGHT", PIXEL, -PIXEL)
			self.PortraitOverlay.Shader = self.PortraitOverlay:CreateTexture(nil, "OVERLAY")
			self.PortraitOverlay.Shader:SetAllPoints(self.PortraitOverlay)
			self.PortraitOverlay.Shader:SetTexture(texture.unitshader)
			self.PortraitOverlay.Shader:SetVertexColor(0, 0, 0, 1)

			self.Portrait.bottomline = CreateFrame("StatusBar", nil, self.Portrait)
			self.Portrait.bottomline:SetStatusBarTexture(texture.blank)
			self.Portrait.bottomline:SetStatusBarColor(0, 0, 0, 1)
			self.Portrait.bottomline:SetHeight(PIXEL)
			self.Portrait.bottomline:SetPoint("BOTTOMLEFT", self.Portrait, "BOTTOMLEFT", 0, -PIXEL)
			self.Portrait.bottomline:SetPoint("BOTTOMRIGHT", self.Portrait, "BOTTOMRIGHT", 0, -PIXEL)
		end

		------------------------------------------------------------
		--		Swing Timer
		------------------------------------------------------------
		if (unit == "player" and db.swingTimer) and ((class == "HUNTER" and db.swingTimerHunter) or not db.swingTimerHunter) then
			self.Swing = CreateFrame("StatusBar", self:GetName().."_Swing", self)
			self.Swing:SetFrameLevel(3)
			self.Swing:SetHeight(3*PIXEL)
			self.Swing:SetPoint("BOTTOMLEFT", self.Portrait, "BOTTOMLEFT", 0, 0)
			self.Swing:SetPoint("BOTTOMRIGHT", self.Portrait, "BOTTOMRIGHT", 0, 0)
			self.Swing:SetStatusBarTexture(texture.normal)
			
			self.Swing:SetStatusBarColor(0.66, 0.66, 0.66)

			self.Swing.bg = self.Swing:CreateTexture(nil, "BORDER")
			self.Swing.bg:SetAllPoints(self.Swing)
			self.Swing.bg:SetTexture(texture.normal)
			self.Swing.bg:SetVertexColor(0.15, 0.15, 0.15)
			
			self.Swing.bottomline = CreateFrame("StatusBar", nil, self.Swing)
			self.Swing.bottomline:SetStatusBarTexture(texture.blank)
			self.Swing.bottomline:SetStatusBarColor(0, 0, 0, 1)
			self.Swing.bottomline:SetFrameStrata(self.Portrait:GetFrameStrata())
			self.Swing.bottomline:SetFrameLevel(self.Portrait:GetFrameLevel() + 1)
			self.Swing.bottomline:SetHeight(PIXEL)
			self.Swing.bottomline:SetPoint("TOPLEFT", self.Swing, "TOPLEFT", 0, PIXEL)
			self.Swing.bottomline:SetPoint("TOPRIGHT", self.Swing, "TOPRIGHT", 0, PIXEL)

			self.Swing.Text = gMedia:makeFontString(self.Swing, numberfont, 12, "OUTLINE")
			self.Swing.Text:SetPoint("LEFT", self.Swing, "LEFT", 0, 0);
			
			self.Swing.disableMelee = db.swingNoMelee;
			self.Swing.diableRanged = db.swingNoRanged;
		end

		------------------------------------------------------------
		--		Unit Names
		------------------------------------------------------------
		if unit == "player" or unit == "target" then
			self.BigName = self:CreateFontString(nil, "OVERLAY")
			self.BigName = gMedia:makeFontString(self.PortraitOverlay, bigfont, fontmultiplier[UI_LAYOUT] * fontheight[20], "OUTLINE")
			self.BigName:SetTextColor(1,1,1)
			self.BigNameInfo = self:CreateFontString(nil, "OVERLAY")
			self.BigNameInfo = gMedia:makeFontString(self.PortraitOverlay, fontname, fontmultiplier[UI_LAYOUT] * fontheight[12], "OUTLINE")
			self.BigNameInfo:SetTextColor(1,1,1)

			if unit == "player" then
				self.BigName:SetPoint("TOPLEFT", self.PortraitOverlay, "TOPLEFT", 4*PIXEL, -4*PIXEL)
				self:Tag(self.BigName, "[NamePortrait]")
				self.BigNameInfo:SetPoint("TOPLEFT", self.BigName, "BOTTOMLEFT", 0, 0)
				self:Tag(self.BigNameInfo, "[level] [race] [raidcolor][smartclass]")
			elseif unit == "target" then
				self.BigName:SetPoint("TOPRIGHT", self.PortraitOverlay, "TOPRIGHT", -4*PIXEL, -4*PIXEL)
				self:Tag(self.BigName, "[NamePortrait]")
				self.BigNameInfo:SetPoint("TOPRIGHT", self.BigName, "BOTTOMRIGHT", 0, 0)
				self:Tag(self.BigNameInfo, "[race] [raidcolor][smartclass] [DiffColor][smartlevel]")
			end
		end

		------------------------------------------------------------
		--		CombatFeedback and PvP status
		------------------------------------------------------------
		if unit == "player" or unit == "target" then
			self.CombatFeedbackText = self:CreateFontString(nil, "OVERLAY")
			self.CombatFeedbackText = gMedia:makeFontString(self.PortraitOverlay, numberfont, fontmultiplier[UI_LAYOUT] * fontheight[22], "OUTLINE")
			self.CombatFeedbackText:SetPoint("CENTER", self.Portrait, "CENTER", 0, 0)
			self.CombatFeedbackText.colors = db.damagecolors
			
			if unit == "player" then
				self.Status = gMedia:makeFontString(self.Portrait, fontname, 24, "OUTLINE")
				self.Status:SetPoint("RIGHT", self.Portrait, "RIGHT", -16*PIXEL, 0)
				self.Status:SetTextColor(0.69, 0.31, 0.31, 0)
				self:Tag(self.Status, "[pvp]")
				
				-- make it a hover event
				self:SetScript("OnEnter", function(self) self.FlashInfo.ManaLevel:Hide() self.Status:SetAlpha(0.5); UnitFrame_OnEnter(self) end)
				self:SetScript("OnLeave", function(self) self.FlashInfo.ManaLevel:Show() self.Status:SetAlpha(0); UnitFrame_OnLeave(self) end)
			end
		end

		------------------------------------------------------------
		--		Castbars
		------------------------------------------------------------
		if not (self:GetAttribute("unitsuffix") == "pet") then
			self.Castbar = CreateFrame("StatusBar", self:GetName().."_Castbar", self)
			self.Castbar:SetStatusBarTexture((unit == "player" or unit == "target") and texture.bars or texture.ace)
			self.Castbar:SetStatusBarColor(1.0, 1.0, 1.0, 0.55)

			-- Only player and target have portraits
			if unit == "player" or unit == "target" then 
				self.Castbar:SetAllPoints(self.Portrait)
				self.Castbar:SetFrameStrata(self.Portrait:GetFrameStrata())
				self.Castbar:SetFrameLevel(self.Portrait:GetFrameLevel() + 1)
			else
				self.Castbar:SetAllPoints(self.Health)
			end

			if unit == "player" or unit == "target" then
				self.Castbar.Icon = self.Castbar:CreateTexture(nil, "ARTWORK")
				self.Castbar.Icon:SetHeight(18*PIXEL)
				self.Castbar.Icon:SetWidth(18*PIXEL)
				self.Castbar.Icon:SetTexCoord(0.1, 0.9, 0.1, 0.9)

				if unit == "player" then
					self.Castbar.Icon:SetPoint("TOPRIGHT", self.Castbar, "TOPLEFT", -8*PIXEL, 0)
				else
					self.Castbar.Icon:SetPoint("TOPLEFT", self.Castbar, "TOPRIGHT", 8*PIXEL, 0)
				end

				self.Castbar.Text = gMedia:makeFontString(self.Castbar, fontname, 11, "OUTLINE")
				self.Castbar.Text:ClearAllPoints()
				self.Castbar.Text:SetWordWrap( false );
				self.Castbar.Text:SetNonSpaceWrap( false );
				self.Castbar.Text:SetWidth((UI_LAYOUT == 0 and 50) or (UI_LAYOUT == 1 and 50) or (UI_LAYOUT == 2 and 90) or (UI_LAYOUT == 3 and 90) or (UI_LAYOUT == 4 and 60) or (UI_LAYOUT == 5 and 60));
				
				self.Castbar.Time = gMedia:makeFontString(self.Castbar, numberfont, 12, "OUTLINE")
				self.Castbar.Time:ClearAllPoints()

				if unit == "player" then 
					self.Castbar.Text:SetPoint("TOPRIGHT", self.Castbar, "TOPRIGHT", -4*PIXEL, -2*PIXEL)
					self.Castbar.Text:SetJustifyH("RIGHT");
					self.Castbar.Time:SetPoint("TOPRIGHT", self.Castbar.Text, "BOTTOMRIGHT", -4*PIXEL, 0)
					self.Castbar.Time:SetJustifyH("RIGHT")
				else
					self.Castbar.Text:SetPoint("TOPLEFT", self.Castbar, "TOPLEFT", 4*PIXEL, -2*PIXEL)
					self.Castbar.Text:SetJustifyH("LEFT");
					self.Castbar.Time:SetPoint("TOPLEFT", self.Castbar.Text, "BOTTOMLEFT", 4*PIXEL, 0)
					self.Castbar.Time:SetJustifyH("LEFT")
				end

				self.Castbar.Text:SetTextColor(1, 1, 1, 0.75)
				self.Castbar.Time:SetTextColor(1, 1, 1, 0.75)
				
				self.Castbar.CustomTimeText = FormatCastbarTime
				self.Castbar.IconBackdrop = gMedia:panel({ parent = self.Castbar, anchor = self.Castbar.Icon, overlay = 0 })
				self.Castbar.IconBackdrop:SetFrameStrata("BACKGROUND")
			end
		end
		------------------------------------------------------------
		--		Power Spark
		------------------------------------------------------------
		if unit == "player" then
			-- Power spark!
			local spark = self.PowerOverlay:CreateTexture(nil, "OVERLAY")
			spark:SetTexture(gMedia.texture["SPARK"])
			spark:SetVertexColor(1, 1, 1, 0.5)
			spark:SetBlendMode("ADD")
			spark:SetHeight(self.PowerOverlay:GetHeight()*2)
			spark:SetWidth(self.PowerOverlay:GetWidth())
			spark.rtl = false
			spark.manatick = false
			spark.highAlpha = 1
			spark.lowAlpha = 0.25
			self.Spark = spark
		end
		------------------------------------------------------------
		--		Overlay Icons
		------------------------------------------------------------
		if not unit or unit == "player" or unit == "target" then
			-- Set up the iconholder frame
			self.IconHolder = CreateFrame("Frame", self:GetName().."_IconHolder", self)
			self.IconHolder:SetAllPoints(self);
			self.IconHolder:SetFrameLevel(129); -- as high as it gets
			local xoff = ((unit == "player" or unit  == "target") and 8*PIXEL) or 4*PIXEL
			local yoff = ((unit == "player" or unit  == "target") and 12*PIXEL) or 8*PIXEL
			local size = ((unit == "player" or unit  == "target") and 18*PIXEL) or 12*PIXEL

			-- Leader
			self.Leader = self.IconHolder:CreateTexture(nil, "OVERLAY")
			self.Leader:SetHeight(size)
			self.Leader:SetWidth(size)
			self.Leader:SetPoint("TOPLEFT", self, "TOPLEFT", -xoff, yoff)
			
			-- Master Looter
			self.MasterLooter = self.IconHolder:CreateTexture(nil, "OVERLAY"); 
			self.MasterLooter:SetHeight(size)
			self.MasterLooter:SetWidth(size)
			self.MasterLooter:SetPoint("TOPRIGHT", self, "TOPRIGHT", xoff, yoff)

			-- Dungeon Role
			if (self:GetParent():GetName():match("oUF_Party")) then
				self.LFDRole = self.IconHolder:CreateTexture(nil, "OVERLAY")
				self.LFDRole:SetHeight(size)
				self.LFDRole:SetWidth(size)
				self.LFDRole:SetPoint("BOTTOMLEFT", self, "BOTTOMLEFT", -xoff, -yoff)
			end
			
			-- Readychecks
			if not unit then
				self.ReadyCheck = self.IconHolder:CreateTexture(nil, "OVERLAY")
				self.ReadyCheck:SetHeight(size)
				self.ReadyCheck:SetWidth(size)
				if (self:GetParent():GetName():match("oUF_Raid")) then
					self.ReadyCheck:SetPoint("BOTTOMLEFT", 13*PIXEL, PIXEL)
				else
					self.ReadyCheck:SetPoint("CENTER", self, "CENTER", 0, 0)
				end
			end
		end


		------------------------------------------------------------
		--		FlashInfo (Low Mana, Resting)
		------------------------------------------------------------
		if unit == "player" then
			self.Combat = self.IconHolder:CreateTexture(nil, "OVERLAY")
			self.Combat:SetHeight(db.iconSize)
			self.Combat:SetWidth(db.iconSize)
			self.Combat:SetPoint("CENTER", self.Health)
			self.Combat:SetTexture([=[Interftexture.ace\CharacterFrame\UI-StateIcon]=])
			self.Combat:SetTexCoord(0.5, 1, 0, 0.421875)

			self.FlashInfo = CreateFrame("Frame", "FlashInfo", self)
			self.FlashInfo:SetScript("OnUpdate", UpdateManaLevel)
			self.FlashInfo.parent = self
			self.FlashInfo:SetToplevel(true)
			self.FlashInfo:SetAllPoints(self.Power)

			self.FlashInfo.ManaLevel = gMedia:makeFontString(self.FlashInfo, fontname, fontmultiplier[UI_LAYOUT] * fontheight[11])
			self.FlashInfo.ManaLevel:SetPoint("CENTER", 0, 0)
			
			if (UnitLevel("player") ~= MAX_PLAYER_LEVEL) then
				self.RestingFrame = CreateFrame("Frame", self:GetName().."_IconHolder_RestingFrame", self.IconHolder)
				self.RestingFrame:SetAllPoints(self.IconHolder)
				self.Resting = self.RestingFrame:CreateTexture(nil, "OVERLAY")
				self.Resting:SetHeight(32*PIXEL)
				self.Resting:SetWidth(32*PIXEL)
				self.Resting:SetPoint("BOTTOMLEFT", self ,"BOTTOMLEFT", -16*PIXEL, -8*PIXEL)
				self.Resting:SetTexture([=[Interftexture.ace\CharacterFrame\UI-StateIcon]=])
				self.Resting:SetTexCoord(0, 0.5, 0, 0.45)
				UIFrameFlash(self.RestingFrame, 1.5, 4.5, 1000000, true, 0, 4.5)
			end
		end

		------------------------------------------------------------
		--		Frame Sizes
		------------------------------------------------------------
		local sizeX, sizeY
		if unit == "player" or unit == "target" then
			sizeX, sizeY = sizes.playerW[UI_LAYOUT], sizes.playerH[UI_LAYOUT]
		elseif unit == "pet" then
			sizeX, sizeY = sizes.playerPetW[UI_LAYOUT], sizes.playerPetH[UI_LAYOUT]
		elseif self:GetAttribute("unitsuffix") == "pet" then
			sizeX, sizeY = sizes.partyPetW[UI_LAYOUT], sizes.partyPetH[UI_LAYOUT]
		elseif self:GetParent():GetName():match("oUF_Party") then
			sizeX, sizeY = sizes.partyW[UI_LAYOUT], sizes.partyH[UI_LAYOUT]
		elseif (unit and unit:find("arena%d")) then
			sizeX, sizeY = sizes.arenaW[UI_LAYOUT], sizes.arenaH[UI_LAYOUT]
		elseif unit == "targettarget" then
			sizeX, sizeY = sizes.targettargetW[UI_LAYOUT], sizes.targettargetH[UI_LAYOUT]
		elseif unit == "focus" then
			sizeX, sizeY = sizes.focusW[UI_LAYOUT], sizes.focusH[UI_LAYOUT]
		elseif unit == "focustarget" then
			sizeX, sizeY = sizes.focustargetW[UI_LAYOUT], sizes.focustargetH[UI_LAYOUT]
		else
			sizeX, sizeY = sizes.defaultW[UI_LAYOUT], sizes.defaultH[UI_LAYOUT]
		end
		sizeX = sizeX * PIXEL
		sizeY = sizeY * PIXEL
		self:SetAttribute("initial-height", sizeY)
		self:SetAttribute("initial-width", sizeX)

		self.RaidIcon = ((unit == "player" or unit == "target") and self.IconHolder:CreateTexture(nil, "OVERLAY") or self.Health:CreateTexture(nil, "OVERLAY"))
		self.RaidIcon:SetHeight(14*PIXEL)
		self.RaidIcon:SetWidth(14*PIXEL)
		self.RaidIcon:SetPoint("TOP", 0, UIPADDING)

		self.Highlight = self:CreateTexture(nil, "HIGHLIGHT")
		self.Highlight:SetAllPoints(self.Health)
		self.Highlight:SetTexture(texture.normal)
		self.Highlight:SetVertexColor(0.84, 0.75, 0.65, 0.15)
		self.Highlight:SetBlendMode("ADD")

		self.DebuffHighlight = self.Health:CreateTexture(nil, "OVERLAY")
		self.DebuffHighlight:SetAllPoints(self.Health)
		self.DebuffHighlight:SetTexture(texture.highlight)
		self.DebuffHighlight:SetVertexColor(0, 0, 0, 0)
		self.DebuffHighlight:SetBlendMode("ADD")
		self.DebuffHighlightAlpha = 1
		self.DebuffHighlightFilter = true

		self.PostUpdateHealth = PostUpdateHealth
		self.PostUpdatePower = PostUpdatePower
		self.PostCreateAuraIcon = CreateAura
		self.PostCreateEnchantIcon = CreateAura
		self.PostUpdateAuraIcon = UpdateAura
		self.PostUpdateEnchantIcons = CreateEnchantTimer
		self.PostUpdateThreat = PostUpdateThreat
		self.OverrideUpdateThreat = OverrideUpdateThreat

		self:SetScale(db.scale)
		if self.Auras then self.Auras:SetScale(db.scale) end
		if self.Buffs then self.Buffs:SetScale(db.scale) end
		if self.Debuffs then self.Debuffs:SetScale(db.scale) end

		HideAura(self)
		return self
	end

	oUF:RegisterStyle("gUI", Styles)
	oUF:SetActiveStyle("gUI")

	------------------------------------------------------------
	--		Spawn and Position Frames
	------------------------------------------------------------
	oUF:Spawn("player", "oUF_player"):SetPoint(spawn.player[UI_LAYOUT].point, spawn.player[UI_LAYOUT].parent, spawn.player[UI_LAYOUT].relativepoint, spawn.player[UI_LAYOUT].xoffset *PIXEL, spawn.player[UI_LAYOUT].yoffset *PIXEL + UNITFRAMES_OFFSET)
	oUF:Spawn("target", "oUF_target"):SetPoint(spawn.target[UI_LAYOUT].point, spawn.target[UI_LAYOUT].parent, spawn.target[UI_LAYOUT].relativepoint, spawn.target[UI_LAYOUT].xoffset *PIXEL, spawn.target[UI_LAYOUT].yoffset *PIXEL + UNITFRAMES_OFFSET)
	oUF:Spawn("pet", "oUF_pet"):SetPoint(spawn.pet[UI_LAYOUT].point, spawn.pet[UI_LAYOUT].parent, spawn.pet[UI_LAYOUT].relativepoint, spawn.pet[UI_LAYOUT].xoffset, spawn.pet[UI_LAYOUT].yoffset)
	oUF:Spawn("focus", "oUF_focus"):SetPoint(spawn.focus[UI_LAYOUT].point, spawn.focus[UI_LAYOUT].parent, spawn.focus[UI_LAYOUT].relativepoint, spawn.focus[UI_LAYOUT].xoffset *PIXEL, spawn.focus[UI_LAYOUT].yoffset *PIXEL)
	oUF:Spawn("focustarget", "oUF_focustarget"):SetPoint(spawn.focustarget[UI_LAYOUT].point, spawn.focustarget[UI_LAYOUT].parent, spawn.focustarget[UI_LAYOUT].relativepoint, spawn.focustarget[UI_LAYOUT].xoffset *PIXEL, spawn.focustarget[UI_LAYOUT].yoffset *PIXEL)
	oUF:Spawn("targettarget", "oUF_targettarget"):SetPoint(spawn.targettarget[UI_LAYOUT].point, spawn.targettarget[UI_LAYOUT].parent, spawn.targettarget[UI_LAYOUT].relativepoint, spawn.targettarget[UI_LAYOUT].xoffset *PIXEL, spawn.targettarget[UI_LAYOUT].yoffset *PIXEL)

	local party = oUF:Spawn("header", "oUF_Party")
	party:SetPoint("TOPLEFT", UIParent, "TOPLEFT", coords.partyX *PIXEL, coords.partyY *PIXEL)
	party:SetManyAttributes(
		"showParty", true,
		"yOffset", -UIPADDING/2,
		"template", "gFrames_Party"
	) 

	party:Hide() --Same as HidePartyFrame() -- hides all party members
			
	--	hide the party frames when in a raid
	local partyisover = CreateFrame("Frame")
	partyisover:RegisterEvent("PLAYER_LOGIN")
	partyisover:RegisterEvent("RAID_ROSTER_UPDATE")
	partyisover:RegisterEvent("PARTY_LEADER_CHANGED")
	partyisover:RegisterEvent("PARTY_MEMBERS_CHANGED")
	partyisover:SetScript("OnEvent", function(self)
		if InCombatLockdown() then
			self:RegisterEvent("PLAYER_REGEN_ENABLED")
		else
			self:UnregisterEvent("PLAYER_REGEN_ENABLED")
			local numraid = GetNumRaidMembers()
			if numraid > 0 then
				party:Hide()
			else
				party:Hide()
			end
		end
	end)
		
	------------------------------------------------------------
	--		Arena Frames
	------------------------------------------------------------
	if not db.noArena then
		local arena = {}
		for i = 1, 5 do
			arena[i] = oUF:Spawn("arena"..i, "oUF_Arena"..i)
			if i == 1 then
				arena[i]:SetPoint(coords.arenaPoint[UI_LAYOUT], UIParent, coords.arenaRelativePoint[UI_LAYOUT], coords.arenaX[UI_LAYOUT] *PIXEL, coords.arenaY[UI_LAYOUT] *PIXEL)
			else
				arena[i]:SetPoint("BOTTOM", arena[i-1], "TOP", 0, UIPADDING)
			end
		end
	end
end




