--[[
	gMedia
	Author: Lars Norberg
	
	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is its designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
]]--

gMedia = CreateFrame("Frame", nil, UIParent);
gMedia.ADDON = "gMedia"; 
gMedia.VERSION = GetAddOnMetadata(gMedia.ADDON,"Version"); 

gMedia:SetScript("OnEvent", function(self, event, ...) self[event](self, ...) end);
gMedia:RegisterEvent("ADDON_LOADED");

-- enUS localization
gMedia.L = {
	-- error messages
	["ERR_LIB_UNKNOWN"] 						= "Library not found: lib['%s']";
	["ERR_MEDIA_UNKNOWN"] 						= "Media not found: lib['%s']['%s']";
	["ERR_MEDIA_EXIST"] 						= "gMedia.lib['%s']['%s'] already exists.";
	["ERR_NOT_STRING"] 							= "'%s' must be a String, got %s.";
	["ERR_USAGE"] 								= "Usage: %s";
	["ERR_CANNOT_DELETE_DEFAULT"] 				= "Cannot unregister default media.";
}

-- simple list of valid points in frames
gMedia.VALIDPOINTS = {
	["TOP"] 				= true;
	["BOTTOM"] 				= true;
	["RIGHT"] 				= true;
	["LEFT"] 				= true;
	["TOPLEFT"] 			= true;
	["TOPRIGHT"] 			= true;
	["BOTTOMLEFT"] 			= true;
	["BOTTOMRIGHT"] 		= true;
	["CENTER"] 				= true;
}

-- simple list of valid layers for textures
gMedia.VALIDLAYERS = {
	["BACKGROUND"] 			= true; -- Level 0. Place the background of your frame here.
	["BORDER"] 				= true; -- Level 1. Place the artwork of your frame here .
	["ARTWORK"] 			= true; -- Level 2. Place the artwork of your frame here.
	["OVERLAY"] 			= true; -- Level 3. Place your text, objects, and buttons in this level.
	["HIGHLIGHT"] 			= true; -- Level 4. Place your text, objects, and buttons in this level. 
}

-- simple list of valid FrameStrata values for frames
gMedia.VALIDFRAMESTRATAS = {
	["PARENT"] 				= true; 
	["BACKGROUND"] 			= true; 
	["LOW"] 				= true; 
	["MEDIUM"] 				= true; 
	["HIGH"] 				= true; 
	["DIALOG"] 				= true; 
	["FULLSCREEN"] 			= true; 
	["FULLSCREEN_DIALOG"] 	= true; 
	["TOOLTIP"] 			= true; 
}

-- main media paths
local path 				= "Interface\\AddOns\\gMedia\\";
local buttonpath 		= "Interface\\AddOns\\gMedia\\buttons\\";
local fontpath 			= "Interface\\AddOns\\gMedia\\fonts\\";
local iconpath 			= "Interface\\AddOns\\gMedia\\icons\\";
local texpath 			= "Interface\\AddOns\\gMedia\\texture\\";

------------------------------------------------------
--	Default library
------------------------------------------------------

-- create the main library table
gMedia.lib = {};

-- shortcut
local lib = gMedia.lib;

-- create the categories
lib.backdrop = {};
lib.background = {};
lib.border = {};
lib.button = {};
lib.color = {};
lib.font = {};
lib.fontn = {};
lib.icon = {};
lib.iconstring = {};
lib.statusbar = {};

-- borders
lib.border["glow"] 							= texpath.."glowtex.tga"; 

-- colors
lib.color["gui"] = {
	["background"] 								= {0.05, 0.05, 0.05, 1};
	["border"] 									= {0.25, 0.25, 0.25, 1};
}
lib.color["RAID_CLASS_COLORS"] = {
	["HUNTER"] 									= { r = 0.67, g = 0.83, b = 0.45 };
	["WARLOCK"] 								= { r = 0.58, g = 0.51, b = 0.79 };
	["PRIEST"] 									= { r = 0.90, g = 0.90, b = 0.90 };
	["PALADIN"] 								= { r = 0.96, g = 0.55, b = 0.73 };
	["MAGE"] 									= { r = 0.41, g = 0.80, b = 0.94 };
	["ROGUE"] 									= { r = 1.00, g = 0.96, b = 0.41 };
	["DRUID"] 									= { r = 1.00, g = 0.49, b = 0.04 };
	["SHAMAN"] 									= { r = 0.00, g = 0.44, b = 0.87 };
	["WARRIOR"] 								= { r = 0.78, g = 0.61, b = 0.43 };
	["DEATHKNIGHT"] 							= { r = 0.77, g = 0.12, b = 0.23 };
}

-- backgrounds
lib.background["default"] 					= texpath.."blank.tga"; 
lib.background["blank"] 					= texpath.."blank.tga"; 
lib.background["fullscreenshader"]			= texpath.."fullscreenshader.tga"; 
lib.background["horizontalshadow"] 			= "Interface\\AchievementFrame\\UI-Achievement-HorizontalShadow.blp"; 
lib.background["satin"] 					= texpath.."satin.tga"; 
lib.background["steel"] 					= texpath.."Steel.tga"; 
lib.background["unitshader"]	 			= texpath.."unitshader.tga"; 
lib.background["vignette"] 					= texpath.."vignette.blp"; 

-- statusbars
lib.statusbar["acebar"]						= texpath.."AceBarFrames.tga"; 
lib.statusbar["bantobar"] 					= texpath.."BantoBar.tga"; 
lib.statusbar["darkbottom"] 				= texpath.."DarkBottom.tga"; 
lib.statusbar["diagonal"] 					= texpath.."diagonalTex.tga"; 
lib.statusbar["highlight"] 					= texpath.."highlightTex.tga"; 
lib.statusbar["minimalist"] 				= texpath.."Minimalist.tga"; 
lib.statusbar["progressbar"] 				= texpath.."progressbar.tga"; 
lib.statusbar["spark"] 						= "Interface\\CastingBar\\UI-CastingBar-Spark"; 

-- buttons
lib.button["button"] 						= buttonpath.."buttonTex.blp"; 
lib.button["gloss"] 						= buttonpath.."gloss.tga"; 
lib.button["flash"] 						= buttonpath.."flash.tga"; 
lib.button["hover"] 						= buttonpath.."hover.tga"; 
lib.button["pushed"]			 			= buttonpath.."pushed.tga"; 
lib.button["checked"] 						= buttonpath.."checked.tga"; 
lib.button["equipped"] 						= buttonpath.."gloss_grey.tga"; 
lib.button["glow"] 							= buttonpath.."UI-ActionButton-Border.blp"; 
lib.button["highlight"] 					= buttonpath.."ButtonHilight-Square.blp"; 

-- fonts
lib.font["arialn"]		 					= fontpath.."ARIALN.ttf"; 
lib.fontn["arialn"]		 					= "Arial Narrow"; 
lib.font["bignoodletitling"] 				= fontpath.."BigNoodleTitling.ttf"; 
lib.fontn["bignoodletitling"] 				= "BigNoodleTitling"; 
lib.font["distinking"] 						= fontpath.."DISTInking-Regular.ttf"; 
lib.fontn["distinking"] 					= "DIST Inking Regular"; 
lib.font["rkillc"] 							= fontpath.."rkillc.ttf"; 
lib.fontn["rkillc"] 						= "Righteous Kill Condensed"; 
lib.font["eurostile-roman"] 				= fontpath.."EuroStile-Roman-DTC.ttf"; 
lib.fontn["eurostile-roman"] 				= "EuroStile-Roman-DTC"; 
lib.font["waukegan-gui"] 					= fontpath.."Waukegan LDO.ttf"; 
lib.fontn["waukegan-gui"] 					= "Waukegan LDO (|cFFFF7D0Ag|r|cFFFFBB00UI|r)"; 
lib.font["gamma"] 							= fontpath.."M025094d.ttf"; 
lib.fontn["gamma"] 							= "MicrogrammaDMedExt"; 
lib.font["steelfis"] 						= fontpath.."steelfis.ttf"; 
lib.fontn["steelfis"] 						= "Steelfish"; 
lib.font["bluecondensed"] 					= fontpath.."Bluecond.ttf";
lib.fontn["bluecondensed"] 					= "Blue Highway Condensed";
lib.font["dreamorphan"] 					= fontpath.."Dreamorp.ttf"; 
lib.fontn["dreamorphan"] 					= "Dream Orphans"; 
lib.font["kenyancoffee"] 					= fontpath.."Kenyc___.ttf"; 
lib.fontn["kenyancoffee"] 					= "Kenyan Coffee"; 
lib.font["zekton"] 							= fontpath.."Zekton__.ttf"; 
lib.fontn["zekton"] 						= "Zekton"; 
lib.font["zrnic"] 							= fontpath.."Zrnic___.ttf"; 
lib.fontn["zrnic"] 							= "Zrnic"; 
lib.font["frizq"] 							= "Fonts\\FRIZQT__.TTF"; 
lib.fontn["frizq"] 							= "Friz Quadrata TT"; 
lib.font["morpheus"] 						= "Fonts\\MORPHEUS.TTF"; 
lib.fontn["morpheus"] 						= "Morpheus"; 
lib.font["skurri"] 							= "Fonts\\SKURRI.TTF"; 
lib.fontn["skurri"] 						= "Skurri"; 

-- icons
lib.icon["alliance"] 						= iconpath.."Alliance.tga"; 
lib.icon["horde"] 							= iconpath.."Horde.tga"; 
lib.icon["bubble"] 							= texpath.."bubbleTex.tga"; 
lib.icon["mailbox"] 						= iconpath.."Mailbox.blp";

-- iconstrings
lib.iconstring["role-tank"] 				= "|TInterface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES.blp:16:16:0:0:64:64:0:19:22:41|t"; 
lib.iconstring["role-heal"] 				= "|TInterface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES.blp:16:16:0:0:64:64:20:39:1:20|t"; 
lib.iconstring["role-dps"] 					= "|TInterface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES.blp:16:16:0:0:64:64:20:39:22:41|t"; 

-- backdrops
lib.backdrop["default"] = {
	backdrop = {
		bgFile = lib.statusbar["minimalist"], 
		edgeFile = lib.background["blank"], 
		edgeSize = 2, 
		insets = {left = 2, right = 2, top = 2, bottom = 2}
	};
	color = {0.15, 0.15, 0.15, 1.0};
	bordercolor = {0, 0, 0, 0.9};
};
lib.backdrop["blank"] = {
	backdrop = {
		bgFile = lib.background["blank"], 
		insets = {left = 0, right = 0, top = 0, bottom = 0}
	};
	color = {0.05, 0.05, 0.05, 1};
	bordercolor = {0.45, 0.45, 0.45, 1};
};
lib.backdrop["blank-inset"] = {
	backdrop = {
		bgFile = lib.background["blank"], 
		edgeFile = lib.background["blank"], 
		edgeSize = 1, 
		tile = false, tileSize = 0,
		insets = {left = -1, right = -1, top = -1, bottom = -1};
	};
	color = {0.05, 0.05, 0.05, 1};
	bordercolor = {0.45, 0.45, 0.45, 1};
};
lib.backdrop["clean"] = {
	backdrop = {
		bgFile = lib.statusbar["diagonal"],
		edgeFile = lib.background["blank"],
		edgeSize = 2, 
		insets = {left = 2, right = 2, top = 2, bottom = 2}
	};
	color = {0.15, 0.15, 0.15, 0.9};
	bordercolor = {0, 0, 0, 0.9};
};
lib.backdrop["glow"] = {
	backdrop = {
		edgeFile = lib.border["glow"];
		edgeSize = 5;
		insets = {left = 2, right = 2, top = 2, bottom = 2};
	};
	color = {0, 0, 0, 1};
	bordercolor = {0, 0, 0, 1};
};
lib.backdrop["satin"] = {
	backdrop = { 
		bgFile = lib.background["satin"], 
		edgeFile = lib.background["blank"], 
		edgeSize =  1,
		insets = {left = -1, right = -1, top = -1, bottom = -1};
	};
	color = lib.color["gui"]["background"]; 
	bordercolor = lib.color["gui"]["border"];
};


------------------------------------------------------
--	Start of gMedia 1.x backward compability
------------------------------------------------------

gMedia.texture = {
	["ACE"] 						= texpath.."AceBarFrames.tga",
	["BANTO"] 						= texpath.."BantoBar.tga",
	["BLANK"] 						= texpath.."blank.tga",
	["BUBBLE"] 						= texpath.."bubbleTex.tga",
	["DARKBOTTOM"] 					= texpath.."DarkBottom.tga",
	["DIAGONAL"] 					= texpath.."diagonalTex.tga",
	["GLOW"] 						= texpath.."glowtex.tga",
	["HIGHLIGHT"] 					= texpath.."highlightTex.tga",
	["LSHADER"] 					= "Interface\\AchievementFrame\\UI-Achievement-HorizontalShadow.blp",
	["MINIMALIST"] 					= texpath.."Minimalist.tga",
	["PROGRESSBAR"] 				= texpath.."progressbar.tga",
	["SATIN"] 						= texpath.."satin.tga",
	["SCREENSHADER"] 				= texpath.."fullscreenshader.tga",
	["SPARK"] 						= "Interface\\CastingBar\\UI-CastingBar-Spark",
	["STEEL"] 						= texpath.."Steel.tga",
	["UNITSHADER"] 					= texpath.."unitshader.tga",
	["VIGNETTE"] 					= texpath.."vignette.blp",
}
gMedia.buttons = {
	["BUTTON"] 						= buttonpath.."buttonTex.blp",
	["NORMAL"] 						= buttonpath.."gloss.tga",
	["FLASH"] 						= buttonpath.."flash.tga",
	["HOVER"] 						= buttonpath.."hover.tga",
	["PUSHED"] 						= buttonpath.."pushed.tga",
	["CHECKED"] 					= buttonpath.."checked.tga",
	["EQUIPPED"] 					= buttonpath.."gloss_grey.tga",
	["BUTTONGLOW"] 					= buttonpath.."UI-ActionButton-Border.blp",
	["BUTTONHIGHLIGHT"] 			= buttonpath.."ButtonHilight-Square.blp",
}
gMedia.icons = {
	["ALLIANCE"] 					= iconpath.."Alliance.tga",
	["HORDE"] 						= iconpath.."Horde.tga",
	["MAILBOX"] 					= iconpath.."Mailbox.blp",
	["ROLETANK"] 					= "|TInterface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES.blp:16:16:0:0:64:64:0:19:22:41|t",
	["ROLEHEAL"] 					= "|TInterface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES.blp:16:16:0:0:64:64:20:39:1:20|t",
	["ROLEDPS"] 					= "|TInterface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES.blp:16:16:0:0:64:64:20:39:22:41|t",
}
gMedia.fonts = {
	["ARIALN"] 						= fontpath.."ARIALN.ttf",
	["RUSSEL"] 						= fontpath.."BigNoodleTitling.ttf",
	["DISTINK"] 					= fontpath.."DISTInking-Regular.ttf",
	["KILL"] 						= fontpath.."rkillc.ttf",
	["EURON"] 						= fontpath.."EuroStile-Roman-DTC.ttf",
	["EURO"] 						= fontpath.."Waukegan LDO.ttf",
	["GAMMA"] 						= fontpath.."M025094d.ttf",
	["STEEL"] 						= fontpath.."steelfis.ttf",
	["BLUEC"] 						= fontpath.."Bluecond.ttf",
	["DREAM"] 						= fontpath.."Dreamorp.ttf",
	["KENYC"] 						= fontpath.."Kenyc___.ttf",
	["ZEKTON"] 						= fontpath.."Zekton__.ttf",
	["ZRNIC"] 						= fontpath.."Zrnic___.ttf",
}
gMedia.backdrops = {
	["DEFAULT"] = {
		backdrop = {
			bgFile = gMedia.texture["MINIMALIST"], 
			edgeFile = gMedia.texture["BLANK"], 
			edgeSize = 2, 
			insets = {left = 2, right = 2, top = 2, bottom = 2}
		};
		color = {0.15, 0.15, 0.15, 1.0};
		bordercolor = {0, 0, 0, 0.9};
	};
	["BLANK"] = {
		backdrop = {
			bgFile = gMedia.texture["BLANK"], 
			insets = {left = 0, right = 0, top = 0, bottom = 0}
		};
		color = {0.05, 0.05, 0.05, 1};
		bordercolor = {0.45, 0.45, 0.45, 1};
	};
	["BLANKINSET"] = {
		backdrop = {
			bgFile = gMedia.texture["BLANK"], 
			edgeFile = gMedia.texture["BLANK"], 
			edgeSize = 1, 
			tile = false, tileSize = 0,
			insets = {left = -1, right = -1, top = -1, bottom = -1};
		};
		color = {0.05, 0.05, 0.05, 1};
		bordercolor = {0.45, 0.45, 0.45, 1};
	};
	["CLEAN"] = {
		backdrop = {
			bgFile = gMedia.texture["DIAGONAL"],
			edgeFile = gMedia.texture["BLANK"],
			edgeSize = 2, 
			insets = {left = 2, right = 2, top = 2, bottom = 2}
		};
		color = {0.15, 0.15, 0.15, 0.9};
		bordercolor = {0, 0, 0, 0.9};
	};
	["GLOW"] = {
		backdrop = {
			edgeFile = gMedia.texture["GLOW"],
			edgeSize = 5,
			insets = {left = 2, right = 2, top = 2, bottom = 2};
		};
		color = {0, 0, 0, 1};
		bordercolor = {0, 0, 0, 1};
	};
	["SATIN"] = {
		backdrop = { 
			bgFile = gMedia.texture["SATIN"], 
			edgeFile = gMedia.texture["BLANK"], 
			edgeSize =  1,
			insets = {left = -1, right = -1, top = -1, bottom = -1};
		};
		color = {0.05, 0.05, 0.05, 1}; 
		bordercolor = {0.25, 0.25, 0.25, 1};
	};
}
gMedia.colors = {
	RAID_CLASS_COLORS = {
		["HUNTER"] = 			{ r = 0.67, g = 0.83, b = 0.45 };
		["WARLOCK"] = 			{ r = 0.58, g = 0.51, b = 0.79 };
		["PRIEST"] = 			{ r = 0.90, g = 0.90, b = 0.90 };
		["PALADIN"] = 			{ r = 0.96, g = 0.55, b = 0.73 };
		["MAGE"] = 				{ r = 0.41, g = 0.80, b = 0.94 };
		["ROGUE"] = 			{ r = 1.00, g = 0.96, b = 0.41 };
		["DRUID"] = 			{ r = 1.00, g = 0.49, b = 0.04 };
		["SHAMAN"] = 			{ r = 0.00, g = 0.44, b = 0.87 };
		["WARRIOR"] = 			{ r = 0.78, g = 0.61, b = 0.43 };
		["DEATHKNIGHT"] = 		{ r = 0.77, g = 0.12, b = 0.23 };
	}
}

------------------------------------------------------
--	End of gMedia 1.x backward compability
------------------------------------------------------



------------------------------------------------------
-- External media management
------------------------------------------------------

-- add media to the library
function gMedia:Register(lib, medianame, mediapath)
	local usage = "gMedia:Register(lib, medianame, mediapath)";

	-- syntax checks
	if type(lib) ~= "string" then 
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("lib", type(lib))..")"); 
	end

	if type(mediapath) ~= "string" then 
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("mediapath", type(mediapath))..")"); 
	end

	if not(self.lib[lib]) then
		error(self.L["ERR_LIB_UNKNOWN"]:format(lib));
	end

	-- if the medianame is nil, fetch the name from the path
	if (medianame == nil) then
		local str = string.reverse(mediapath)
		medianame = string.reverse(string.sub(str, 1, (string.find(str, "\\"))-1))
	end

	if (type(medianame) ~= "string") then	
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("medianame", type(medianame))..")"); 
	end

	lib = lib:lower();

	if self.lib[lib][medianame] then 
		error(self.L["ERR_MEDIA_EXIST"]:format(lib, medianame));
	end
	
	-- add the media, nothing more to it
	self.lib[lib][medianame] = mediapath;
end

-- remove media from the library
function gMedia:Unregister(lib, medianame)
	local usage = "gMedia:Unregister(lib, medianame)";

	-- syntax checks
	if type(lib) ~= "string" then 
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("lib", type(lib))..")"); 
	end
	if type(medianame) ~= "string" then	
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("medianame", type(medianame))..")"); 
	end
	if not(self.lib[lib]) then
		error(self.L["ERR_LIB_UNKNOWN"]:format(lib));
	end
	if not(self.lib[lib][medianame]) then
		error(self.L["ERR_MEDIA_UNKNOWN"]:format(lib, medianame));
	end
	
	lib = lib:lower();

	-- protect our default library
	error(self.defaultlib[lib][medianame], self.L["ERR_CANNOT_DELETE_DEFAULT"])
	
	-- remove the media
	self.lib[lib][medianame] = nil;
end

-- creates a truetable of our default media to protect it from being deleted
function gMedia:RegisterDefaults()
	self.defaultlib = self.defaultlib or {}
	for i,v in pairs(self.lib) do
		self.defaultlib[i] = self.defaultlib[i] or {}
		for key,val in pairs(self.lib[i]) do
			self.defaultlib[i][key] = true;
		end
	end
end

-- get some media
-- not strictly needed if you know the direct link
-- advantages of this function is it's error reports
function gMedia:Get(lib, medianame)
	local usage = "gMedia:Get(lib, medianame)";

	-- syntax checks
	if type(lib) ~= "string" then 
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("lib", type(lib))..")"); 
	end
	if type(medianame) ~= "string" then	
		error(self.L["ERR_USAGE"]:format(usage).." - ("..self.L["ERR_NOT_STRING"]:format("medianame", type(medianame))..")"); 
	end

	lib 			= lib:lower();
	
	if not(self.lib[lib]) then
		error(self.L["ERR_LIB_UNKNOWN"]:format(lib));
	end
	if not(self.lib[lib][medianame]) then
		error(self.L["ERR_MEDIA_UNKNOWN"]:format(lib, medianame));
	end
	
	return self.lib[lib][medianame];
end


------------------------------------------------------
--	Functions
------------------------------------------------------
function gMedia:SetFont()
end
function gMedia:SimpleTexture(args)
	if not(type(args)) == "table" then return end
	
	local tex = {};

	-- attribute					uservalue												defaults 
	tex.frame 					= args.frame 											or UIParent;
	tex.texture 				= args.texture 											or self.lib.background["default"];
	tex.layer 					= (self.VALIDLAYERS[args.layer]) and args.layer 		or "BACKGROUND";
	tex.name 					= args.name 											or nil;
	tex.blend 					= args.blend 											or "BLEND";
	tex.alpha 					= args.alpha 											or 1;
	tex.texcoord 				= args.texcoord 										or {0, 1, 0, 1};
	
	local t = tex.frame:CreateTexture(tex.name, tex.layer);
	t:SetTexture(tex.texture);
	t:SetBlendMode(tex.blend);
	t:SetAlpha(tex.alpha);
	t:SetTexCoord(unpack(tex.texcoord));
	t:SetAllPoints(tex.frame);
	
end

function gMedia:formatTime(seconds)
	if not(seconds) then return "0.0"; end
	local day, hour, minute = 86400, 3600, 60
	if seconds >= day then
		return format("%dd", floor(seconds/day + 0.5)), seconds % day
	elseif seconds >= hour then
		return format("%dh", floor(seconds/hour + 0.5)), seconds % hour
	elseif seconds >= minute then
		if seconds <= minute * 5 then
			return format("%d:%02d", floor(seconds/60), seconds % minute), seconds - floor(seconds)
		end
		return format("%dm", floor(seconds/minute + 0.5)), seconds % minute
	elseif seconds >= minute / 12 then
		return floor(seconds + 0.5), (seconds * 100 - floor(seconds * 100))/100
	end
	return format("%.1f", seconds), (seconds * 100 - floor(seconds * 100))/100
end
function gMedia:shortValue(value)
	if not(value) then return "0.0"; end
	if value >= 1e6 then
		return ("%.1fm"):format(value / 1e6):gsub("%.?0+([km])$", "%1")
	elseif value >= 1e3 or value <= -1e3 then
		return ("%.1fk"):format(value / 1e3):gsub("%.?0+([km])$", "%1")
	else
		return value
	end
end

function gMedia:ADDON_LOADED()
	if not(arg1 == self.ADDON) then return end
	
	-- We're loaded, no need to watch for this anymore
	self:UnregisterEvent("ADDON_LOADED");
	self:SetScript("OnEvent", nil);

	-- build a list of our defaults
	self:RegisterDefaults();

end

--
-- 	IMPORTANT!
-- 	Do not call these function before PLAYER_LOGIN(), as certain information is not available to the client before that!
--
function gMedia:pixel()
	-- routine to figure out exactly how big a pixel is in the current resolution and UIScale
	return (768 / tonumber(string.match(GetCVar("gxResolution"), "%d+x(%d+)"))) / tonumber(GetCVar("uiScale"))
end

-- panel expects an effective framescale of 1 on the parent object, or the pixel borders won't be pixels
function gMedia:panel(args)	
	local panel = {};
	local PIXEL = gMedia:pixel();	

	-- attribute					uservalue					defaults 									description
	panel.parent 				= args.parent 				or UIParent 								-- parent to our panel
	panel.anchor	 			= args.anchor 				or panel.parent 					 		-- frame to anchor our panel to
	panel.bottomright 			= args.bottomright 			or panel.anchor 					 		-- bottomright corner to anchor to, overrides anchor
	panel.topleft 				= args.topleft 				or panel.anchor 							-- topleft corner to anchor to, overrides anchor
	panel.overlay 				= args.overlay 				or 1										-- 1 to use overlay, 0 to skip it
	panel.overlaytexture 		= args.overlaytexture 		or gMedia.lib.background["satin"]			-- overlay texture 
	panel.overlaycolor 			= args.overlaycolor 		or {0.35, 0.35, 0.35, 1} 					-- color of the overlay
	panel.alpha 				= args.alpha 				or 1										-- alpha of the backdrop
	panel.backdropcolor 		= args.backdropcolor 		or gMedia.lib.backdrop["satin"].color		-- color of the background
	panel.bordercolor 			= args.bordercolor 			or gMedia.lib.backdrop["satin"].bordercolor	-- color of the border
	panel.framestrata 			= args.framestrata 			or "BACKGROUND"								-- framestrata of the entire object
	panel.padding 				= args.padding				or 3										-- padding between the border and the contents
	panel.paddingx 				= args.paddingx				or 0 										-- horizontal padding. added to general padding.
	panel.paddingy 				= args.paddingy 		 	or 0 										-- vertical padding. added to general padding.
	panel.offsettop 			= args.offsettop 		 	or 0 										-- top padding. 
	panel.offsetbottom 			= args.offsetbottom 	 	or 0 										-- bottom padding. 
	panel.offsetleft 			= args.offsetleft 		 	or 0 										-- left padding. 
	panel.offsetright			= args.offsetright 		 	or 0 										-- right padding. 
	panel.shadow 				= args.shadow 				or 1										-- 1 to use shadows behind the panel, 0 to not

	-- create the frame for the border
	local frame = CreateFrame("Frame", panel.parent:GetName() and panel.parent:GetName().."_panel_border" or nil, panel.parent);

	-- create the 2 first parts of our 3px border
	frame:SetPoint("TOPLEFT", panel.topleft, "TOPLEFT", -(panel.padding + panel.paddingx + panel.offsetleft)/PIXEL, (panel.padding + panel.paddingy + panel.offsettop)/PIXEL );
	frame:SetPoint("BOTTOMRIGHT", panel.bottomright, "BOTTOMRIGHT", (panel.padding + panel.paddingx + panel.offsetright)/PIXEL, -(panel.padding + panel.paddingy + panel.offsetbottom)/PIXEL);
	frame:SetBackdrop(gMedia.lib.backdrop["blank-inset"].backdrop);
	frame:SetBackdropColor(unpack(panel.backdropcolor));
	frame:SetBackdropBorderColor(unpack(panel.bordercolor));
	frame:SetFrameStrata(panel.framestrata);
	frame:SetFrameLevel(max(1, panel.parent:GetFrameLevel() - 2));
	frame:SetScale(PIXEL);
	
	-- create the frame for the backdrop, and the 3rd part of our 3px border
	if panel.overlay == 1 then
		--   set a minimum of 2px for the padding, or the border will be invisible
		local overlaypadding = max(2, panel.padding - 2);
		frame.bg = frame:CreateTexture(frame:GetName() and frame:GetName().."_overlay" or nil, "ARTWORK", frame);
		frame.bg:ClearAllPoints();
		frame.bg:SetPoint("TOPLEFT", frame, "TOPLEFT", (overlaypadding)/PIXEL, -(overlaypadding)/PIXEL);
		frame.bg:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -(overlaypadding)/PIXEL, (overlaypadding)/PIXEL);
		frame.bg:SetTexture(panel.overlaytexture);
		frame.bg:SetVertexColor(unpack(panel.overlaycolor));
	end
	
	-- create a shadow
	if panel.shadow == 1 then
		frame.shadow = self:shadow({ parent = frame, name = frame:GetName() and frame:GetName().."_shadow" or nil});
	end
	
	frame:SetAlpha(panel.alpha);
	
	return frame
end
function gMedia:shadow(args)
	local shadow = {};
	local PIXEL = gMedia:pixel();	

	-- attribute					uservalue					defaults 									description
	shadow.parent 				= args.parent 				or args.anchor or UIParent					-- parent to our panel
	shadow.anchor 				= args.anchor 				or shadow.parent 							-- what frame to anchor the shadow to
	shadow.name	 				= args.name 				or nil								  		-- name of our frame

	local frameshadow = CreateFrame("Frame", shadow.name, shadow.parent)
	frameshadow:SetFrameStrata(shadow.anchor:GetFrameStrata())
	frameshadow:SetFrameLevel(shadow.anchor:GetFrameLevel()-1)
	frameshadow:SetBackdrop(gMedia.lib.backdrop["glow"].backdrop)
	frameshadow:SetBackdropBorderColor(0.05, 0.05, 0.05, 0.85)
	frameshadow:SetPoint("TOPLEFT", shadow.anchor, "TOPLEFT", -4.5/PIXEL, 4.5/PIXEL)
	frameshadow:SetPoint("BOTTOMRIGHT", shadow.anchor, "BOTTOMRIGHT", 4.5/PIXEL, -4.5/PIXEL)
	frameshadow:SetScale(PIXEL)

	return frameshadow
end

-- font
function gMedia:makeFontString(parent, fontname, fontHeight, fontStyle)
	local fs = parent:CreateFontString(nil, "OVERLAY")
	fs:SetFont(fontname, fontHeight, fontStyle)
	fs:SetJustifyH("LEFT")
	fs:SetShadowColor(0, 0, 0)
	fs:SetShadowOffset(1.25, -1.25)
	return fs
end


