--[[
	gTip
	Author: Lars Norberg
	
	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is its designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
]]--

local L = gLib:FetchLocale("gTip", GetLocale());

local db = {
	color = {
		str1 = "|cff%02x%02x%02x";
		str2 = "|r";
		realm = { r = 1.0; g = 1.0; b = 1.0; };
		guild = { r = 1.0; g = 1.0; b = 0.7; };
	};
	tooltips = {
		-- dropdown lists and menus
		DropDownList1Backdrop;
		DropDownList2Backdrop;
		DropDownList1MenuBackdrop;
		DropDownList2MenuBackdrop;
		
		-- normal tooltips
		GameTooltip; 
		ItemRefTooltip; 
		ShoppingTooltip1; 
		ShoppingTooltip2; 
		ShoppingTooltip3;
		WorldMapTooltip; 
		
		-- chatframe menus
		ChatMenu;
		EmoteMenu;
		LanguageMenu;
		VoiceMacroMenu;
		
		-- the title picker frame
		PlayerTitlePickerFrame;
	};
}

local defaults = {
	-- 0 = default pos, 1 = always to mouse, 2 = only anchor units
	anchortocursor = 2;
	
	-- 0 = always show, 1 = hide in combat
	hidewhilecombat = 0;
	
	-- show player titles
	showtitle = 1;
	
	-- show realm names for other realms
	showrealm = 1;
	
	-- show health values on the health bars
	showhealth = 1;
}

local slash = {
	[1] = {
		name = "menu";
		func = function() 
			InterfaceOptionsFrame_OpenToCategory(gTip.optionspanel.name); 
		end;
		slashes = {"gtip"};
	};
}

gTip_DB = gTip_DB or {}

gTip = gLib:New({
	name = "gTip";
	userDB = gTip_DB;
	defaultDB = defaults;
	slash = slash;
})

local NAME 		= gTip.NAME;
local CHARNAME 	= gTip.CHARNAME;
local REALM 	= gTip.REALM;
local VERSION 	= gTip.VERSION;

local _G = _G;
local pairs, unpack = pairs, unpack;

local backdrop 			= gMedia.lib.backdrop["satin"];
local emptybackdrop 	= gMedia.lib.backdrop["blank-inset"];
local PIXEL 			= function() return gMedia:pixel() end

local function UnitColor(unit)
	local color = { r = 0.6, g = 0.6, b = 0.6 };
	local happiness = {
		[1] = { r = 0.79, g = 0.25, b = 0.25};
		[2] = { r = 0.89, g = 0.69, b = 0.25};
		[3] = { r = gMedia.lib.color.RAID_CLASS_COLORS["HUNTER"].r * 0.66, g = gMedia.lib.color.RAID_CLASS_COLORS["HUNTER"].g * 0.66, b = gMedia.lib.color.RAID_CLASS_COLORS["HUNTER"].b * 0.66};
	}

	if unit then
		-- class colored unit
		if UnitIsPlayer(unit) then
			color = gMedia.lib.color.RAID_CLASS_COLORS[(select(2, UnitClass(unit)))]

		-- it's your pet, show it's happiness
		elseif unit == "pet" and GetPetHappiness() then
			color = happiness[(select(1, GetPetHappiness()))]
		else
			color = FACTION_BAR_COLORS[UnitReaction(unit, "player")]
		end
	end
	
    return color.r, color.g, color.b
end
local function UnitTip(self)
	local lines = self:NumLines();
	local name, unit = self:GetUnit();

	
	if unit then
		if gTip_DB.showtitle and (gTip_DB.showtitle == 1) then
			local pvpname = UnitPVPName(unit);
			if pvpname then 
				name = pvpname;
			end
		end
		
		local level = UnitLevel(unit);
		local levelColor = GetQuestDifficultyColor(level);

		if level == -1 then
			level = "??"
			levelColor = { r = 1.0, g = 0.0, b = 0.0 }
		end

		local r, g, b = UnitColor(unit) 
		
		_G["GameTooltipTextLeft1"]:SetFormattedText(db.color.str1.."%s"..db.color.str2, r*255, g*255, b*255, name)
		
		if UnitIsPlayer(unit) then
			local race = UnitRace(unit)
			local name, realm = UnitName(unit)
			
			if realm then 
				local txt = db.color.str1.."%s"..db.color.str2..db.color.str1.." - %s"..db.color.str2
				_G["GameTooltipTextLeft1"]:SetFormattedText(txt, r*255, g*255, b*255, name, db.color.realm.r*255, db.color.realm.g*255, db.color.realm.b*255, realm);
			end


			if GetGuildInfo(unit) then
				_G["GameTooltipTextLeft2"]:SetFormattedText(db.color.str1.."<%s>"..db.color.str2, db.color.guild.r*255, db.color.guild.g*255, db.color.guild.b*255, GetGuildInfo(unit))
			end

			local n = GetGuildInfo(unit) and 3 or 2

			if GetCVar("colorblindMode") == "1" then n = n + 1 end
			_G["GameTooltipTextLeft"..n]:SetFormattedText(db.color.str1.."%s"..db.color.str2.." %s", levelColor.r*255, levelColor.g*255, levelColor.b*255, level, race);

		else

			local classification = UnitClassification(unit)
			local creatureType = UnitCreatureType(unit)

			classification = (classification == "rareelite" and "+ (rare)") or (classification == "rare" and " (rare)") or (classification == "elite" and "+") or "";

			for i = 2, lines do
				local line = _G["GameTooltipTextLeft"..i]
				if not line or not line:GetText() then return end
				if (level and line:GetText():find("^"..LEVEL)) or (creatureType and line:GetText():find("^"..creatureType)) then
					line:SetFormattedText(db.color.str1.."%s%s"..db.color.str2.." %s", levelColor.r*255, levelColor.g*255, levelColor.b*255, level, classification, creatureType or "")
					break
				end
			end
		end

		-- target of target
		if UnitExists(unit.."target") and unit~="player" then
			local r, g, b = UnitColor(unit.."target")
			local name, realm = UnitName(unit.."target")
			GameTooltip:AddLine(((name == CHARNAME) and ((realm == nil) or (realm == REALM))) and "|cFFFF0000>> YOU <<|r" or ">> "..name.." <<", r, g, b)
		end

		-- set a minimum width for unit our tooltips
		if (self:GetWidth() < 100) then
			self:SetWidth(100);
		end
	end
end
local function Update(self, ...)
end
local function Position(tooltip, parent)
	local x 		= (MultiBarLeft:IsShown() and 37 or 0) + (MultiBarRight:IsShown() and 37 or 0);
	local y 		= ( (not(IsAddOnLoaded("gUI")) and MultiBarBottomRight:IsShown()) and 37 or 0) + (IsAddOnLoaded("gUI") and 60 or 0);
	
	if (gTip_DB.anchortocursor == 0) then
		tooltip:SetOwner(parent,"ANCHOR_NONE");
		tooltip.default = 1;
		tooltip:ClearAllPoints();
		tooltip:SetPoint("BOTTOMRIGHT", "UIParent" ,"BOTTOMRIGHT", -(8 + x), 110 + y);

	elseif (gTip_DB.anchortocursor == 1) then
		tooltip:ClearAllPoints();
		tooltip:SetOwner(parent, "ANCHOR_CURSOR");
		tooltip.default = 1;

	elseif (gTip_DB.anchortocursor == 2) then
		if UnitExists("mouseover") then
			tooltip:ClearAllPoints();
			tooltip:SetOwner(parent, "ANCHOR_CURSOR");
			tooltip.default = 1;
		else
			tooltip:SetOwner(parent,"ANCHOR_NONE");
			tooltip.default = 1;
			tooltip:ClearAllPoints();
			tooltip:SetPoint("BOTTOMRIGHT", "UIParent" ,"BOTTOMRIGHT", -(8 + x), 110 + y);
		end
	end
end
local function showIcon()
	-- pointer to our frame
	local frame = _G["ItemLinkIcon"]

	-- hide the old frame while we set up things
	frame:Hide()

	-- is there an icon for us to display?
	local link = (select(2, _G["ItemRefTooltip"]:GetItem()));
	local icon = link and GetItemIcon(link);

	-- nah, just leave
	if not icon then return end
	
	-- yes! show it!
	frame.icon:SetTexture(icon);
	frame:Show();
end

function gTip:OnInit()
	self:ArgumentCheck( gTip_DB, defaults);
end
function gTip:OnEnable()
	-- set the global for tooltip visibility in combat
	SHOW_TOOLTIPS_IN_COMBAT = ((gTip_DB.hidewhilecombat == 1) and false) or true;

	-- set up our tooltips and other frames
	-- also register the event to hide it in combat here, as this needs to be dynamic and re-registered
	for i,v in pairs(db.tooltips) do
		v:SetBackdrop(backdrop.backdrop)
		v:SetScript("OnShow", function(self) 
			self:SetBackdropColor(backdrop.color[1], backdrop.color[2], backdrop.color[3], 0.8); 
			self:SetBackdropBorderColor(unpack(backdrop.bordercolor)); 
			self:RegisterEvent("PLAYER_REGEN_ENABLED");
			self:RegisterEvent("PLAYER_REGEN_DISABLED");

			-- hide the tip if in combat
			if ((gTip_DB.hidewhilecombat == 1) and InCombatLockdown()) then 
				self:Hide(); 
				self:RegisterEvent("PLAYER_REGEN_ENABLED"); 
			end
		end)
		v:SetScript("OnEvent", function(self, event) 
			-- hide the tip when entering combat
			if (event == "PLAYER_REGEN_DISABLED") and (gTip_DB.hidewhilecombat == 1) then 
				self:Hide(); 
				self:RegisterEvent("PLAYER_REGEN_ENABLED"); 
			end
			
			-- display it when leaving combat
			if (event == "PLAYER_REGEN_ENABLED") then 
				self:Show(); 
				self:RegisterEvent("PLAYER_REGEN_DISABLED"); 
			end
		end)
		v:SetScale(PIXEL()); 
	end

	-- create our itemlink icon frame, complete with a button and all
	local icon = CreateFrame("Frame", "ItemLinkIcon", _G["ItemRefTooltip"]);
	icon:SetPoint("TOPRIGHT", _G["ItemRefTooltip"], "TOPLEFT", -3, 2);
	icon:SetHeight(32);
	icon:SetWidth(32);

	icon.icon = icon:CreateTexture("icon", "ARTWORK");
	icon.icon:SetPoint("TOPLEFT", icon, "TOPLEFT", 3, -3);
	icon.icon:SetPoint("BOTTOMRIGHT", icon, "BOTTOMRIGHT", -3, 3);
	icon.icon:SetTexCoord(.05, .95, .05, .95);

	icon.button = icon:CreateTexture("border", "OVERLAY");
	icon.button:SetAllPoints(icon);
	icon.button:SetTexture(gMedia.lib.button["gloss"]);
	icon.button:SetVertexColor(unpack(backdrop.bordercolor));

	-- modify the tooltip healthbars to our liking
	GameTooltipStatusBar:ClearAllPoints();
	GameTooltipStatusBar:SetPoint("TOP", GameTooltip, "TOP", -1, -5); -- ("TOPLEFT", GameTooltip, "BOTTOMLEFT", -1, -5);
	GameTooltipStatusBar:SetPoint("TOPLEFT", GameTooltip, "TOPRIGHT", 1, -5); -- ("TOPRIGHT", GameTooltip, "BOTTOMRIGHT", 1, -5);
	GameTooltipStatusBar:SetStatusBarTexture( gMedia.lib.statusbar["bantobar"] ); -- gMedia.lib.statusbar["bantobar"]
	GameTooltipStatusBar:SetStatusBarColor(0.3, 0.9, 0.3, 1); -- 0.3, 0.9, 0.3, 1
	GameTooltipStatusBar:SetHeight(6);
	GameTooltipStatusBar.Text = GameTooltipStatusBar:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall");
	GameTooltipStatusBar.Text:Place("CENTER", GameTooltipStatusBar , "CENTER", 0, 2);
	GameTooltipStatusBar.Text:SetFont((GameTooltipStatusBar.Text:GetFont()), 12, "THINOUTLINE");
	GameTooltipStatusBar:HookScript("OnUpdate", function(self, elapsed)
		if not(self:IsShown()) then return end
		self.elapsed = (self.elapsed or 0) + elapsed
		if (self.elapsed >= 0.1) then
			if (gTip_DB.showhealth == 1) then
				local value, minValue, maxValue = GameTooltipStatusBar:GetValue(), GameTooltipStatusBar:GetMinMaxValues()
				if (value > 0) then
					self.Text:SetText(gTip:Value(value).."/"..gTip:Value(maxValue));
					self:SetBackdropColor(emptybackdrop.color[1], emptybackdrop.color[2], emptybackdrop.color[3], 0.8); -- 0.8
					self:SetBackdropBorderColor(emptybackdrop.color[1], emptybackdrop.color[2], emptybackdrop.color[3], 0.8); -- 0.8
				else
					self.Text:SetText("");
					self:SetBackdropColor(0,0,0,0);
					self:SetBackdropBorderColor(0,0,0,1);
				end
			else
				self.Text:SetText("");
			end
			self.elapsed = self.elapsed - 0.1;
		end
	end)
	GameTooltipStatusBar:SetBackdrop(emptybackdrop.backdrop)
	GameTooltipStatusBar:SetBackdropColor(emptybackdrop.color[1], emptybackdrop.color[2], emptybackdrop.color[3], 0); -- 0.8
	GameTooltipStatusBar:SetBackdropBorderColor(unpack(emptybackdrop.bordercolor)); 
	
	--GameTooltipStatusBar:Hide() -- experiemental!

	-- hook our tooltip functions
	GameTooltip:HookScript("OnTooltipSetUnit", UnitTip)
	GameTooltip:HookScript("OnUpdate", Update)
	hooksecurefunc("GameTooltip_SetDefaultAnchor", Position)
	hooksecurefunc("SetItemRef", showIcon);

	-- Setup Interface Options panel
	self.optionspanel = self:CreateOptionsPanel({
		title = NAME;
		version = VERSION;
		panel = ((IsAddOnLoaded("gUI") and gUI.optionspanel) or nil);
		option = {
			[10] = {
				name = L["gTip_OPTIONS_ANCHORMODE"];
				type = "dropdown";
				value = gTip_DB.anchortocursor;
				list = {
					{
						level 		= 1;
						text 		= function() 
							return ((gTip_DB.anchortocursor == 0) and L["gTip_OPTIONS_NOANCHOR_OPTION"]) or ((gTip_DB.anchortocursor == 1) and L["gTip_OPTIONS_ANCHORTOMOUSE_OPTION"]) or ((gTip_DB.anchortocursor == 2) and L["gTip_OPTIONS_ANCHORSMART_OPTION"]) or "";
						end;
						menulist = {
							{
								text 			= L["gTip_OPTIONS_NOANCHOR"];
								notCheckable 	= 1;
								func = function() 
									gTip_DB.anchortocursor = 0;
								end;
							};
							{
								text 			= L["gTip_OPTIONS_ANCHORTOMOUSE"];
								notCheckable 	= 1;
								func = function() 
									gTip_DB.anchortocursor = 1;
								end;
							};
							{
								text 			= L["gTip_OPTIONS_ANCHORSMART"];
								notCheckable 	= 1;
								func = function() 
									gTip_DB.anchortocursor = 2;
								end;
							};
						};
					};
				};
			};
			[30] = {
				name = L["gTip_OPTIONS_HIDEINCOMBAT"];
				type = "button";
				value = (gTip_DB.hidewhilecombat == 1) and true or false;
				panelfunction = function() gTip_DB.hidewhilecombat = (gTip_DB.hidewhilecombat == 1) and 0 or 1; 
					SHOW_TOOLTIPS_IN_COMBAT = ((gTip_DB.hidewhilecombat == 1) and false) or true;
				end;
			};
			[40] = {
				name = L["gTip_OPTIONS_SHOWTITLES"];
				type = "button";
				value = (gTip_DB.showtitle == 1) and true or false;
				panelfunction = function() gTip_DB.showtitle = (gTip_DB.showtitle == 1) and 0 or 1; end;
			};
			[50] = {
				name = L["gTip_OPTIONS_SHOWREALM"];
				type = "button";
				value = (gTip_DB.showrealm == 1) and true or false;
				panelfunction = function() gTip_DB.showrealm = (gTip_DB.showrealm == 1) and 0 or 1; end;
			};
			[60] = {
				name = L["gTip_OPTIONS_SHOWHEALTH"];
				type = "button";
				value = (gTip_DB.showhealth == 1) and true or false;
				panelfunction = function() gTip_DB.showhealth = (gTip_DB.showhealth == 1) and 0 or 1; end;
			};
		};
	}, gTip_DB, defaults);
end

