--[[
	gBags
	Author: Lars Norberg
	
	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is its designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
]]--

local L = gLib:FetchLocale("gBags", GetLocale());

---
-- Bag category filter names. do NOT change these!!
-- They are used for item filtering, not display!
---

L.filter = {};
L.filter.Weapon, L.filter.Armor, L.filter.Container, L.filter.Consumables, L.filter.Glyph, L.filter.Trades, L.filter.Projectile, L.filter.Quiver, L.filter.Recipe, L.filter.Gem, L.filter.Misc, L.filter.Quest = GetAuctionItemClasses();

gBags_DB = gBags_DB or {}

local db = {
	framestrata 		= "HIGH";
	framelevel 			= 60;

	buttonsize 			= 27; -- gUI button size
	padding 			= 2;
	
	scale 				= 1.0; 

	-- buttons we need to adopt
	buttons = {
		bags = {
			bagbuttons = {
				--MainMenuBarBackpackButton;
				CharacterBag0Slot;
				CharacterBag1Slot;
				CharacterBag2Slot;
				CharacterBag3Slot;
			};
			keyringbutton = KeyRingButton;
		};
		bank = {
			bankbuttons = {
				BankFrameBag1;
				BankFrameBag2;
				BankFrameBag3;
				BankFrameBag4;
				BankFrameBag5;
				BankFrameBag6;
				BankFrameBag7;
			};
			bankpurchasebutton = {
				BankFramePurchaseInfo;
			};
		};
	};
	
	-- items we want in different categories
	extra = {
		keys = {
			[18250] 		= true; -- Gordok Shackle Key
			[30688] 		= true; -- Deathforge Key
			[45798] 		= true; -- Heroic Celestial Planetarium Key
		};
		misc = {
		};
		gizmos = {
			[18606] 		= true; -- Alliance Battle Standard
			[31666] 		= true; -- Battered Steam Tonk Controller
			[49917] 		= true; -- Brazie's Gnomish Pleasure Device
			[34686] 		= true; -- Brazier of Dancing Flames
			[33927] 		= true; -- Brewfest Pony Keg
			[49704] 		= true; -- Carved Ogre Idol
			[37710] 		= true; -- Crashin' Thrashin' Racer Controller
			[23767] 		= true; -- Crashin' Thrashin' Robot
			[38301] 		= true; -- D.I.S.C.O.
			[36863] 		= true; -- Decahedral Dwarven Dice
			[30542] 		= true; -- Dimensional Ripper - Area 52
			[18984] 		= true; -- Dimensional Ripper - Everlook
			[37863] 		= true; -- Direbrew's Remote
			[21540] 		= true; -- Elune's Lantern
			[18232] 		= true; -- Field Repair Bot 74A
			[34113] 		= true; -- Field Repair Bot 110G
			[33223] 		= true; -- Fishing Chair
			[40772] 		= true; -- Gnomish Army Knife
			[18645] 		= true; -- Gnomish Alarm-o-Bot
			[18587] 		= true; -- Goblin Jumper Cables XL
			[18258] 		= true; -- Gordok Ogre Suit
			[40110] 		= true; -- Haunted Memento
			[45631] 		= true; -- High-powered Flashlight
			[43499] 		= true; -- Iron Boot Flask
			[52251] 		= true; -- Jaina's Locket
			[49040] 		= true; -- Jeeves
			[40768] 		= true; -- MOLL-E
			[52201] 		= true; -- Muradin's Favor
			[35275] 		= true; -- Orb of the Sin' dorei
			[34499] 		= true; -- Paper Flying Machine Kit
			[49703] 		= true; -- Perpetual Purple Firework
			[32566] 		= true; -- Picnic Basket
			[34480] 		= true; -- Romantic Picnic Basket
			[40769] 		= true; -- Scrapbot Construction Kit
			[19045] 		= true; -- Stormpike Battle Standard
			[52253] 		= true; -- Sylvanas' Music Box
			[38578] 		= true; -- The Flag of Ownership
			[43824] 		= true; -- The Schools of Arcane Magic - Mastery
			[44430] 		= true; -- Titanium Seal of Dalaran
			[44606] 		= true; -- Toy Train Set
			[45984] 		= true; -- Unusual Compass
			[18986] 		= true; -- Ultrasafe Transporter: Gadgetzan
			[30544] 		= true; -- Ultrasafe Transporter: Toshley's Station
			[34068] 		= true; -- Weighted Jack-o'-Lantern
			[45057] 		= true; -- Wind-Up Train Wrecker
			[17712] 		= true; -- Winter Veil Disguise Kit
			[18660] 		= true; -- World Enlarger
			[48933] 		= true; -- Wormhole Generator: Northrend
			[36862] 		= true; -- Worn Troll Dice
			[23821] 		= true; -- Zapthrottle Mote Extractor

		};
	};
	
	font 				= gMedia.lib.font["zrnic"];
	fontsize 			= 12;
	fontcolor 			= UI_PANEL_VALUECOLOR or {1, 0.8, 0.2};

	backdrop 			= gMedia.lib.backdrop["satin"];
	color 				= gMedia.lib.color["gui"]["background"]; 
	bordercolor 		= gMedia.lib.color["gui"]["border"];
	bankcolor 			= {0.35, 0.35, 0.75, 1};
	overlaycolor 		= {0.35, 0.35, 0.35, 1};
	
	texture = {
		blank 				= gMedia.lib.background["blank"];
		gloss 				= gMedia.lib.button["gloss"];
		hover 				= gMedia.lib.button["hover"];
		pushed 				= gMedia.lib.button["pushed"];
		checked 			= gMedia.lib.button["checked"];
		glow 				= gMedia.lib.button["glow"];
		highlight	 		= gMedia.lib.button["highlight"];
		progressbar 		= gMedia.lib.statusbar["progressbar"];
	};
}
local defaults = {
	-- 1 to lock the bags, 0 to remain movable
	locked 			= 0;

	-- always hide soul shards
	hideshards 		= 1;
	
	-- always hide soul shards
	hidejunk 		= 0;

	-- always hide ammo
	hideammo 		= 1;
	
	-- scale
	scale 			= 1.0;
	
	-- default positions of the bags and bank
	bagspot	= { 
		point 		= "BOTTOMRIGHT";
		anchor 		= "UIParent";
		relpoint 	= "BOTTOMRIGHT";
		x 			= -12;
		y 			= 220;
	};
	bankspot = { 
		point 		= "TOPLEFT"; 
		anchor 		= "UIParent";
		relpoint 	= "TOPLEFT";
		x 			= 12;
		y 			= -12; 
	};
}

-- main addon object
gBags = gLib:newAddon("gBags");
gBags:RegisterEvent("EQUIPMENT_SETS_CHANGED");
gBags:Slash( "gBagsMenu", function(msg, editbox) 
	local command, rest = msg:match("^(%S*)%s*(.-)$");
	if (command:lower() == "scale") then
		rest = tonumber(rest) or 0;
		if (rest >= 1.0) and (rest <= 2.0) then
			gBags_DB.scale = rest;
			db.scale = rest;
			gBags:ScaleAll();
		else
			print(L["ERR_VALID_SCALES"]:format(1,2));
		end
	else
		InterfaceOptionsFrame_OpenToCategory(gBags.optionspanel.name); 
	end
end, 
{"gbags"});

local _G = _G;
local abs, format, pairs, strfind, unpack = abs, format, pairs, strfind, unpack;

local ADDON 		= gBags.ADDON;
local NAME 			= gBags.NAME;
local VERSION 		= gBags.VERSION;
local noop 			= gBags.noop;

local Bags 			= cargBags:NewImplementation("gBags");
local Container 	= Bags:GetContainerClass();
local Button 		= Bags:GetItemButtonClass();

local BagButton 	= cargBags.classes["BagButton"];



-- replaced blizzard functions
BankFrameItemButtonBag_OnClick = function(self, button) 
	local inventoryID = self:GetInventorySlot();
	local hadItem = PutItemInBag(inventoryID);
	local id = self:GetID();
	UpdateBagButtonHighlight(id);
end
BagSlotButton_OnClick = function(self)
	local id = self:GetID();
	local translatedID = id - CharacterBag0Slot:GetID() + 1;
	local hadItem = PutItemInBag(id);
	BagSlotButton_UpdateChecked(self);
end

-- cache the Blizzard EQ manager sets
local function cacheEQSets()
	db.eq = db.eq or {};
	wipe(db.eq);
	for s = 1, GetNumEquipmentSets() do
		local set = GetEquipmentSetItemIDs(GetEquipmentSetInfo(s))
		for i,v in pairs(set) do
			if v then 
				db.eq[v] = true;
			end
		end
	end
end

-- returns the real amount of equipped ammo
local function ammoCount()
	local slot = GetInventorySlotInfo("AmmoSlot");
	local count = GetInventoryItemCount("player", slot);
	local icon = GetInventoryItemTexture("player", slot);
	
	-- empty slots are returned as 1, so we need an extra check here
	if ((count == 1) and (not icon)) then
		count = 0;
	end
	return count or 0, icon or ""
end

-- Container functions
function Container:OnCreate(name, settings)
	self.Settings = settings;

	self:EnableMouse(true);

	if strfind(name, "Bank") then
		self.panel = gMedia:panel({ parent = self; overlaycolor = db.bankcolor; });
	else
		self.panel = gMedia:panel({ parent = self; overlaycolor = db.overlaycolor; });
	end

	self.panel:SetAlpha(0.9);
	self.panel:ClearAllPoints();
	self.panel:SetPoint("TOPLEFT", self, "TOPLEFT", 4, -4);
	self.panel:SetPoint("BOTTOMRIGHT", self, "BOTTOMRIGHT", -4, 4);
	self.panel:SetScale( 1 / db.scale );
	
	settings.Columns = settings.Columns or 8;
	settings.Scale = settings.Scale or 1;
	settings.Parent = settings.Parent or Bags;

	self:SetParent(settings.Parent);
	self:SetScale(settings.Scale * db.scale);
	
	self:SetFrameStrata(db.framestrata);
	self:SetFrameLevel(db.framelevel);

	if settings.Name then
		local title = self:CreateFontString(nil, "OVERLAY");
		title:SetFont(db.font, db.fontsize);
		title:Place("TOPLEFT", self, "TOPLEFT", 12, -9);
		title:SetText("|cFF"..(format("%02x%02x%02x", db.fontcolor[1]*255, db.fontcolor[2]*255, db.fontcolor[3]*255))..settings.Name.."|r");
	end

	if name == "Main" or name == "Bank" then
		self:SetMovable(true)
		self:RegisterForClicks("LeftButton", "RightButton");
		self:SetScript("OnMouseDown", function(self)
			if not(gBags_DB.locked == 1) then
				self:ClearAllPoints(); 
				self:StartMoving();
			end
		end)
		self:SetScript("OnMouseUp", function(self) 
			if not(gBags_DB.locked == 1) then
				self:StopMovingOrSizing();
			end
		end);

		local infoFrame = CreateFrame("Button", nil, self);
		infoFrame:SetPoint("BOTTOMLEFT", 16, 3);
		infoFrame:SetPoint("BOTTOMRIGHT", -16, 3);
		infoFrame:SetHeight(32);

		local spaceholder = CreateFrame("Frame", nil, infoFrame);
		
		local space = spaceholder:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall");
		space:Place("BOTTOMLEFT", infoFrame, "BOTTOMLEFT", 3, 6);
		space:SetJustifyH("LEFT");
		space:SetFont(db.font, floor(db.fontsize * 1.5));
		
		spaceholder:RegisterEvent("BAG_UPDATE");

		if (name == "Bank") then
			space:SetText(string.tag("[free:bankframe+bank]/[max:bankframe+bank]"));
			spaceholder:RegisterEvent("BANKFRAME_OPENED");
			spaceholder:SetScript("OnEvent", function() 
				space:SetText(string.tag("[free:bankframe+bank]/[max:bankframe+bank]"));
			end);
		elseif (name == "Main") then
			space:SetText(string.tag("[free:backpack+bags]/[max:backpack+bags]"));
			spaceholder:SetScript("OnEvent", function() 
				space:SetText(string.tag("[free:backpack+bags]/[max:backpack+bags]"));
			end);
		end

		local text = infoFrame:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall");
		text:Place("TOPLEFT", space, "TOPRIGHT", 2, 0);
		text:SetText(L["free"]);

		local spacegraph = CreateFrame("StatusBar", nil, infoFrame);
		spacegraph:SetBackdrop({ bgFile = db.texture.progressbar });
		spacegraph:SetBackdropColor(1, 1, 1, 0.2);
		spacegraph:Size(80,20);
		spacegraph:Place("BOTTOMLEFT", space, "BOTTOMRIGHT", 32, 2);

		spacegraph.bar = spacegraph:CreateTexture(nil, "OVERLAY");
		spacegraph.bar:Size(80,20);
		spacegraph.bar:Place("TOPLEFT", spacegraph, "TOPLEFT", 0, 0);
		spacegraph.bar:SetTexture(db.texture.progressbar);

		local bags = ((name == "Main") and { 0, 1, 2, 3, 4 }) or ((name == "Bank") and { -1, 5, 6, 7, 8, 9, 10, 11 }) or {};
		local function setFree()
			local free, total = 0, 0;
			for _,i in pairs( bags ) do
				free, total, used = free + GetContainerNumFreeSlots(i), total + GetContainerNumSlots(i), total - free;
			end
			free = free/total;
			spacegraph.bar:SetWidth(((free > 0) and free or 0.0001) * 80);
			spacegraph.bar:SetTexCoord(0,0, 0,1, free,0, free,1);
		end
		setFree();

		spacegraph:SetScript("OnEvent", function() setFree(); end);
		spacegraph:RegisterEvent("BAG_UPDATE");
		if (name == "Bank") then
			spacegraph:RegisterEvent("BANKFRAME_OPENED");
		end
		
		local money = CreateFrame("Frame", nil, self);
		money.text = money:CreateFontString(nil, "OVERLAY");
		money.text:SetFont(db.font, db.fontsize);
		money.text:Place("TOPRIGHT", self, "TOPRIGHT", -18, -9);
		money.text:SetText("|cFFFFFFFF"..("[money]"):tag().."|r");

		money:SetScript("OnEvent", function(self) self.text:SetText("|cFFFFFFFF"..("[money]"):tag().."|r"); end)
		money:RegisterEvent("PLAYER_MONEY");
		money:RegisterEvent("PLAYER_ENTERING_WORLD");
		money:RegisterEvent("PLAYER_LOGIN");

		if (name == "Main") then
			-- Plugin: SearchBar
			-- If we specify a frame as an optional arg #2, then this frame
			-- shows the search on click at its own place
			local search = self:SpawnPlugin("SearchBar", infoFrame)
			
			-- make this a global search, for all bags
			search.isGlobal = true;

			self.tokenframe = CreateFrame("Frame", self:GetName().."_TokenFrame", self);
			self.tokenframe:SetPoint("TOPLEFT", self, "TOPLEFT", 16, -28);
			self.tokenframe:SetPoint("BOTTOMRIGHT", self, "TOPRIGHT", -16, -43);
		
			self.tokenframe.text = self.tokenframe:CreateFontString(nil, "OVERLAY");
			self.tokenframe.text:SetFont(db.font, floor(db.fontsize * 0.75));
			self.tokenframe.text:SetWidth(self.tokenframe:GetWidth());
			self.tokenframe.text:Place("CENTER", self.tokenframe, "CENTER", 0, 0);

			-- create a global pointer to the tokenframe
			db.tokenframe = self.tokenframe;

			-- function to set or update the tokentext
			local function updatetokens()
				local faction 		= (UnitFactionGroup("player"));
				local soulshard 	= GetItemCount(6265, false, false);
				local bankshard 	= max(GetItemCount(6265, true, false) - soulshard, 0);
				local ammo, aicon 	= ammoCount();
				local class 		= (select(2, UnitClass("player")));
				local text 			= "";

				-- warlock soul shards, displays shards in the bank in brackets, if any
				if (class == "WARLOCK") then
					text = text.."|cFF"..((soulshard > 0) and "FFFFFF" or "666666")..soulshard..((bankshard > 0) and "("..bankshard..")" or "").."|r  |TInterface\\ICONS\\INV_Misc_Gem_Amethyst_02:14:14:0:0:64:64:5:59:5:59|t";
				end
				
				-- toggle the ammoframe
				-- this is useful for everybody, not just Hunters, since nobody really wants to see the ammo
				if (gBags_DB.hideammo == 1) then
					if ((ammo ~= 0) and (Bags:GetContainer("Main_Ammo"):IsShown())) then 
						Bags:GetContainer("Main_Ammo"):Hide(); 
					end
					if ((ammo == 0) and not(Bags:GetContainer("Main_Ammo"):IsShown())) then 
						Bags:GetContainer("Main_Ammo"):Show(); 
					end
				end

				-- only display the ammo count if you're a Hunter
				-- I might add an option to toggle this
				if (class == "HUNTER") then	
					text = text.."|cFF"..((ammo > 0) and "FFFFFF" or "666666")..ammo.."|r  |T"..aicon..":14:14:0:0:64:64:5:59:5:59|t";
				end
				
				-- watched currencies, if any
				for i = 1, GetNumWatchedTokens()  do
					local name, count, extraCurrencyType, icon = GetBackpackCurrencyInfo(i);
					if count then
						local tcount = count
						icon = "  |T"..icon..":14:14:0:0:64:64:5:59:5:59|t";
						if extraCurrencyType == 1 then 
							icon = "  |TInterface\\PVPFrame\\PVP-ArenaPoints-Icon:14:14:0:0:64:64:0:64:0:64|t";
						elseif extraCurrencyType == 2 then 
							icon = "  |T"..gMedia.lib.icon[strlower(faction)]..":14:14:0:0:64:64:0:64:0:64|t";
						end
						tcount = "|cFF"..((count > 0) and "FFFFFF" or "666666")..tcount.."|r"
						tracked = tcount..icon;
						if text == "" then 
							text = tracked;
						else
							text = text.."    "..tracked;
						end
					end
				end
				db.tokenframe.text:SetText(text);
				Bags:GetContainer("Main"):OnContentsChanged();
			end
		
			-- make sure our tokendisplay updates when it should
			self.tokenframe:SetScript("OnEvent", function(self) updatetokens(); end)
			self.tokenframe:RegisterEvent("BAG_UPDATE");
			self.tokenframe:RegisterEvent("CURRENCY_DISPLAY_UPDATE");
			self.tokenframe:RegisterEvent("KNOWN_CURRENCY_TYPES_UPDATE");
			self.tokenframe:RegisterEvent("UNIT_INVENTORY_CHANGED");

			-- this is so it updates our tokendisplay if the bags are open when changing tracked currencies
			hooksecurefunc("TokenFrame_Update", updatetokens);

			updatetokens();

			-- BagBar
			self.bagbar = self:SpawnPlugin("BagBar", "bags", self);
			self.bagbar:Place("TOPRIGHT", self, "BOTTOMRIGHT", -6, -4);
			self.bagbar:SetFrameStrata(db.framestrata);
			self.bagbar:SetFrameLevel(db.framelevel + 30);
			self.bagbar:Hide();

			local width, height = self.bagbar:LayoutButtons("grid", 4, 4, 2, -2);
			self.bagbar:SetWidth(width + 4 + db.buttonsize/2 + 4);
			self.bagbar:SetHeight(height + 4);

			local panel = gMedia:panel({ parent = self.bagbar; overlaycolor = db.overlaycolor; padding = 2; framestrata = db.framestrata });
			panel:SetAlpha(0.9);

			-- create a keyring button
			local keyringbutton = CreateFrame("CheckButton", nil, self.bagbar);
			keyringbutton:Size(db.buttonsize/2, db.buttonsize);
			keyringbutton:Place("TOPRIGHT", self.bagbar, "TOPRIGHT", -2, -2);
			keyringbutton:SetCheckedTexture("Interface\\Buttons\\CheckButtonHilight", "ADD");
			keyringbutton:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD");
			keyringbutton:RegisterForClicks("LeftButtonUp");
			keyringbutton.Texture = keyringbutton:CreateTexture();
			keyringbutton.Texture:SetTexture('Interface/Buttons/UI-Button-KeyRing');
			keyringbutton.Texture:SetAllPoints(keyringbutton);
			keyringbutton.Texture:SetTexCoord(0, 0.5625, 0, 0.609375);
			keyringbutton:SetScript("OnClick", function() 
				local keyring = Bags:GetContainer("Keyring");
				if keyring:IsShown() then 
					keyring:Hide();
				else
					keyring:Show();
				end
			end);
			
			-- create a clickable button to toggle the bagbar
			self.bagbar.textholder = CreateFrame("Button", self:GetName().."_BagButton", infoFrame);
			self.bagbar.text = infoFrame:CreateFontString(nil, "OVERLAY");
			self.bagbar.text:SetFont(db.font, db.fontsize);
			self.bagbar.text:Place("BOTTOMRIGHT", infoFrame, "BOTTOMRIGHT", -3, 6);
			self.bagbar.text:SetText(L["BagBar"]);
			self.bagbar.textholder:SetAllPoints(self.bagbar.text);
			self.bagbar.textholder:SetScript("OnClick", function() 
				if self.bagbar:IsShown() then
					self.bagbar:Hide();
				else
					self.bagbar:Show();
				end
			end)
		end

		-- Bank bagbar
		if (name == "Bank") then
			-- BagBar
			self.bagbar = self:SpawnPlugin("BagBar", "bank", self);
			self.bagbar:Place("TOPRIGHT", self, "BOTTOMRIGHT", -6, -4);
			self.bagbar:SetFrameStrata(db.framestrata);
			self.bagbar:SetFrameLevel(db.framelevel + 30);
			self.bagbar:Hide();

			local width, height = self.bagbar:LayoutButtons("grid", 7, 4, 2, -2);
			self.bagbar:SetWidth(width + 4);
			self.bagbar:SetHeight(height + 4);

			local panel = gMedia:panel({ parent = self.bagbar; overlaycolor = db.overlaycolor; padding = 2; framestrata = db.framestrata });
			panel:SetAlpha(0.9);

			-- create a clickable button to toggle the bagbar
			self.bagbar.textholder = CreateFrame("Button", self:GetName().."_BagButton", infoFrame);
			self.bagbar.text = infoFrame:CreateFontString(nil, "OVERLAY");
			self.bagbar.text:SetFont(db.font, db.fontsize);
			self.bagbar.text:Place("BOTTOMRIGHT", infoFrame, "BOTTOMRIGHT", -3, 6);
			self.bagbar.text:SetText(L["BagBar"]);
			self.bagbar.textholder:SetAllPoints(self.bagbar.text);
			self.bagbar.textholder:SetScript("OnClick", function() 
				if self.bagbar:IsShown() then
					self.bagbar:Hide();
				else
					self.bagbar:Show();
				end
			end)

			-- purchase slots
			self.purchase = CreateFrame("Frame", self:GetName().."_PurchaseSlots", infoFrame);
			
			BankFramePurchaseButton:SetParent(self.purchase);
			BankFramePurchaseButton:SetFrameStrata(db.framestrata);
			BankFramePurchaseButton:SetFrameLevel(db.framelevel + 10);
			BankFramePurchaseButton:Place("RIGHT", self.bagbar.text, "LEFT", -36, 2);
			BankFramePurchaseButton:SetScript("OnClick", function(self) 
				PlaySound("igMainMenuOption");
				StaticPopup_Show("CONFIRM_BUY_BANK_SLOT");
			end)
			
			local numSlots, full = GetNumBankSlots();
			local cost = GetBankSlotCost(numSlots);
			local purchasecolor
			BankFrame.nextSlotCost = cost;

			if( GetMoney() >= cost ) then
				purchasecolor = "|cFFFFFFFF";
			else
				purchasecolor = "|cFFFF0000";
			end			
			self.purchase.text = self.purchase:CreateFontString(nil, "OVERLAY");
			self.purchase.text:SetFont(db.font, db.fontsize);
			self.purchase.text:Place("RIGHT", BankFramePurchaseButton, "LEFT", -12, 0);
			self.purchase.text:SetText(purchasecolor..COSTS_LABEL.."  "..("[money:"..cost.."]"):tag().."|r");
			
			self.purchase:RegisterEvent("PLAYERBANKBAGSLOTS_CHANGED");
			self.purchase:SetScript("OnEvent", function(self) 
				local numSlots, full = GetNumBankSlots();
				local cost = GetBankSlotCost(numSlots);
				local purchasecolor
				BankFrame.nextSlotCost = cost;

				if( GetMoney() >= cost ) then
					purchasecolor = "|cFFFFFFFF";
				else
					purchasecolor = "|cFFFF0000";
				end	
				self.text:SetText(purchasecolor..COSTS_LABEL.."  "..("[money:"..cost.."]"):tag().."|r");
				if (full) then
					self:Hide();
				end
				UpdateBagSlotStatus();
			end)

			if (full) then
				self.purchase:Hide();
			end
		end
	end
end
function Container:Scale()
	self:SetScale(self.Settings.Scale * db.scale);
	self.panel:SetScale( 1 / db.scale );
end
function Container:OnContentsChanged()
	
	local name = self.name;
	local width, height;
	local class = (select(2, UnitClass("player")));
	local classextra = 0;
	local num = GetNumWatchedTokens();
	if ( name == "Main" ) and (( num > 0) or ( class == "WARLOCK" ) or ( class == "HUNTER" )) then
		if (( class == "WARLOCK" ) or ( class == "HUNTER" )) then
			classextra = 20;
		else
			local name, count, extraCurrencyType, icon = GetBackpackCurrencyInfo(num);
			if count then
				classextra = 20;
			end
		end
	end

	-- sort the buttons the smart way
	self:SortButtons("smartSort");

	-- update the size
	width, height = self:LayoutButtons("grid", self.Settings.Columns, 4, 12, -(29 + classextra));

	if name == "Main" or name == "Bank" then
		self:SetSize(width + 24, height + 68 + classextra);
	else
		-- a little magic with the sizes, plus modifying the alpha, combined with frames having 0 padding between them,
		-- results in "smart" placing of the frames ;)
		if (height <= 0) then
			self:SetSize(width + 24, 0.001);
			self:SetAlpha(0);
		else
			self:SetSize(width + 24, height + 40);
			self:SetAlpha(1);
		end
	end
end
function Container.layouts:grid(columns, spacing, xOffset, yOffset)
	columns, spacing = columns or 8, spacing or 5
	xOffset, yOffset = xOffset or 0, yOffset or 0
	local width, height = (db.buttonsize), (db.buttonsize);
	local col, row = 0, 0;
	for i, button in ipairs(self.buttons) do
		col = i % columns
		if(col == 0) then col = columns end
		row = math.ceil(i/columns)

		local xPos = (col-1) * (width + spacing)
		local yPos = -1 * (row-1) * (height + spacing)

		button:Place("TOPLEFT", self, "TOPLEFT", xPos+xOffset, yPos+yOffset)
	end

	return columns * (width+spacing)-spacing, row * (height+spacing)-spacing
end
function Container.sort.smartSort(a, b)
	local aID, bID = GetContainerItemID(a.bagID, a.slotID) or 0, GetContainerItemID(b.bagID, b.slotID) or 0
	local aR, bR = aID and (select(3, GetItemInfo(aID))) or 0, bID and (select(3, GetItemInfo(bID))) or 0
	if aR == bR then
		local aL, bL = aID and (select(4, GetItemInfo(aID))) or 1, bID and (select(4, GetItemInfo(bID))) or 1
		if aL == bL then
			if aID == bID then
				local aCount, bCount = (select(2,GetContainerItemInfo(a.bagID, a.slotID))) or 1, (select(2,GetContainerItemInfo(b.bagID, b.slotID))) or 1
				return aCount > bCount
			else
				return aID > bID
			end
		else
			return aL > bL
		end
	else
		return aR > bR
	end
end

-- Button functions
local slotsNum = 0
function Button:Create(tpl)
	slotsNum = slotsNum+1
	local name = "cargBagsSlot"..slotsNum

	local button = setmetatable(CreateFrame("Button", name, nil, tpl), self.__index)

	button:Size(db.buttonsize);

	button.Icon 		= _G[name.."IconTexture"];
	button.Count 		= _G[name.."Count"];
	button.Cooldown 	= _G[name.."Cooldown"];
	button.Quest 		= _G[name.."IconQuestTexture"];
	button.Border 		= _G[name.."NormalTexture"];

	button:SetHighlightTexture(db.texture.hover);
	button:SetPushedTexture(db.texture.pushed);
	button:SetNormalTexture(db.texture.gloss);

	button.Border:ClearAllPoints();
	button.Border:SetPoint("TOPLEFT", button, "TOPLEFT", -2, 2);
	button.Border:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", 2, -2);

	-- trim the icon to fit into our buttons
	button.Icon:SetTexCoord(0.05, 0.95, 0.05, 0.95);
	button.Icon:ClearAllPoints();
	button.Icon:SetPoint("TOPLEFT", button.Border, "TOPLEFT", 4, -4);
	button.Icon:SetPoint("BOTTOMRIGHT", button.Border, "BOTTOMRIGHT", -4, 4);
	
	-- position the quest/rarity glow
	button.Quest:ClearAllPoints();
	button.Quest:SetPoint("TOPLEFT", button.Border, "TOPLEFT", 2, -2);
	button.Quest:SetPoint("BOTTOMRIGHT", button.Border, "BOTTOMRIGHT", -2, 2);
	
	-- fix the item count
	button.Count:SetFont(select(1, button.Count:GetFont()), floor(0.5 + (db.fontsize / 1.2)), "THINOUTLINE");
	button.Count:ClearAllPoints();
	button.Count:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", -2, 4);
	button.Count:SetDrawLayer("OVERLAY");

	if(button.OnCreate) then button:OnCreate(tpl) end

	button.glowAlpha = 0.65
	button.glowBlend = "ADD"
	button.glowCoords = { 16/64, 48/64, 16/64, 48/64 }
	
	return button
end
function Button:UpdateQuest(item)
	local r,g,b,a 			= 1.0, 1.0, 1.0, 1.0;
	local tL,tR,tT,tB 		= 0.05, 0.95, 0.05, 0.95;
	local blend 			= "BLEND";
	local texture

	if(item.questID and not item.questActive) then
		texture = TEXTURE_ITEM_QUEST_BANG
	elseif(item.questID or item.isQuestItem) then
		texture = TEXTURE_ITEM_QUEST_BORDER
	elseif(item.rarity and item.rarity > 1 and self.glowTex) then
		a, r,g,b = self.glowAlpha, GetItemQualityColor(item.rarity)
		texture = self.glowTex
		blend = self.glowBlend
		tL,tR,tT,tB = unpack(self.glowCoords)
	end

	if(texture) then
		self.Quest:SetTexture(texture)
		self.Quest:SetTexCoord(tL,tR,tT,tB)
		self.Quest:SetBlendMode(blend)
		self.Quest:SetVertexColor(r,g,b,a)
		self.Quest:Show()
	else
		self.Quest:Hide()
	end

	if(self.OnUpdateQuest) then self:OnUpdateQuest(item) end
end
function Button:OnUpdate(item)
	-- color the border based on bag type
	local bagType = (select(2, GetContainerNumFreeSlots(self.bagID)));
	
	-- keyring
	if(bagType == 256) then
		self.Border:SetVertexColor(1, 0.7, 0.1, 1);
	
	-- ammo / soulshards
	elseif(bagType and (bagType > 0 and bagType < 8)) then
		self.Border:SetVertexColor(0.85, 0.85, 0.35, 1);
		
	-- profession bags
	elseif(bagType and bagType > 4) then
		self.Border:SetVertexColor(0.1, 0.65, 0.1, 1);
	
	-- normal bags
	else
		self.Border:SetVertexColor(unpack(db.bordercolor));
	end
end

-- BagButton functions
local buttonNum = 0
function BagButton:Create(bagID)
	buttonNum = buttonNum+1
	local name = "cargBagsBagButton"..buttonNum

	local button = setmetatable(CreateFrame("CheckButton", name, nil, "ItemButtonTemplate"), self.__index)

	local invID = ContainerIDToInventoryID(bagID)
	button.invID = invID
	button:SetID(invID)
	button.bagID = bagID

	button:RegisterForDrag("LeftButton", "RightButton")
	button:RegisterForClicks("anyUp")

	button:SetSize(db.buttonsize, db.buttonsize)
	
	button.Icon = 		_G[name.."IconTexture"]
	button.Count = 		_G[name.."Count"]
	button.Cooldown = 	_G[name.."Cooldown"]
	button.Quest = 		_G[name.."IconQuestTexture"]
	button.Border =		_G[name.."NormalTexture"]

	button:SetHighlightTexture(db.texture.hover);
	button:SetPushedTexture(db.texture.pushed);
	button:SetNormalTexture(db.texture.gloss);
	button:SetCheckedTexture(db.texture.checked, "ADD")

	button.Border:ClearAllPoints();
	button.Border:SetPoint("TOPLEFT", button, "TOPLEFT", -2, 2);
	button.Border:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", 2, -2);

	-- as ugly hacks go, this one is a winner
	button.Border:SetVertexColor(unpack(db.bordercolor));
	button.Border.SetVertexColor = noop;

	-- trim the icon to fit into our buttons
	button.Icon:SetTexCoord(0.05, 0.95, 0.05, 0.95);
	button.Icon:ClearAllPoints();
	button.Icon:SetPoint("TOPLEFT", button.Border, "TOPLEFT", 4, -4);
	button.Icon:SetPoint("BOTTOMRIGHT", button.Border, "BOTTOMRIGHT", -4, 4);

	if button.Quest then
		-- position the quest/rarity glow
		button.Quest:ClearAllPoints();
		button.Quest:SetPoint("TOPLEFT", button.Border, "TOPLEFT", 2, -2);
		button.Quest:SetPoint("BOTTOMRIGHT", button.Border, "BOTTOMRIGHT", -2, 2);
	end
	
	-- fix the item count
	button.Count:SetFont(select(1, button.Count:GetFont()), floor(0.5 + (db.fontsize / 1.2)), "THINOUTLINE");
	button.Count:Place("BOTTOMRIGHT", button, "BOTTOMRIGHT", -2, 4);
	button.Count:SetDrawLayer("OVERLAY");

	cargBags.SetScriptHandlers(button, "OnClick", "OnReceiveDrag", "OnEnter", "OnLeave", "OnDragStart")

	if(button.OnCreate) then button:OnCreate(bagID) end

	return button
end

-- Bag functions
function Bags:OnInit()

	local INVERTED = -1 -- with inverted filters (using -1), everything goes into this bag when the filter returns false

	local hideAmmo 				= function(item) return not(item.type and (item.type == L.filter.Projectile)) end
	local hideAmmoBags 			= function(item) 
		local bagType = (select(2, GetContainerNumFreeSlots(item.bagID))); 
		return not ( bagType and (bagType > 0 and bagType < 8) ) 
	end
	local hideEmpty 			= function(item) return item.texture ~= nil end
	local hideJunk 				= function(item) return not item.rarity or item.rarity > 0 end
	local hideSoulShards 		= function(item) 
		local itemID = GetContainerItemID(item.bagID, item.slotID);
		return not (itemID and (itemID == 6265))
	end
	local onlyAmmo 				= function(item) return item.type and (item.type == L.filter.Projectile) end
	local onlyRecipe 			= function(item) return item.type and (item.type == L.filter.Recipe) end
	local onlyGizmos 			= function(item) 
		local itemID = GetContainerItemID(item.bagID, item.slotID);
		return (itemID and (db.extra.gizmos[itemID]))
	end
	local onlyMisc 				= function(item) 
		local itemID = GetContainerItemID(item.bagID, item.slotID);
		return (item.type and (item.type == L.filter.Misc)) or (itemID and (db.extra.misc[itemID]))
	end
	local onlyArmor 			= function(item) return item.type and (item.type == L.filter.Armor) end 
	local onlyBags 				= function(item) return item.bagID >= 0 and item.bagID <= 4 end
	local onlyBank 				= function(item) return item.bagID == -1 or item.bagID >= 5 and item.bagID <= 11 end
	local onlyConsumables 		= function(item) return item.type and (item.type == L.filter.Consumables) end
	local onlyEpics 			= function(item) return item.rarity and item.rarity > 3 end
	local onlyGems 				= function(item) return item.type and (item.type == L.filter.Gem) end
	local onlyGlyphs 			= function(item) return item.type and (item.type == L.filter.Glyph) end
	local onlyItemSets 			= function(item) 
		local itemID = GetContainerItemID(item.bagID, item.slotID);
		return itemID and (db.eq[itemID])
	end
	local onlyJunk 				= function(item) return item.rarity and (item.rarity == 0) end
	local onlyKeyring 			= function(item) 
		local itemID = GetContainerItemID(item.bagID, item.slotID);
		return (item.bagID == -2) or (itemID and (db.extra.keys[itemID]))
	end
	local onlyQuest 			= function(item) return item.type and (item.type == L.filter.Quest) end
	local onlyRareEpics 		= function(item) return item.rarity and item.rarity > 3 end
	local onlySoulShards 		= function(item) 
		local itemID = GetContainerItemID(item.bagID, item.slotID);
		return itemID and (itemID == 6265) 
	end
	local onlyTradeGoods 		= function(item) return item.type and (item.type == L.filter.Trades) end
	local onlyWeapon 			= function(item) return item.type and (item.type == L.filter.Weapon) end
	local nothing 				= function(item) return end

	

	-- note: the order of these categories are vital, do not change them, or items will appear in wrong categories!!
	
	local banksets = Container:New("Bank_Sets", { Bags = "bankframe+bank"; Name = L["Sets"]; Columns = 8; });
		banksets:SetFilter(onlyBank, true);
		banksets:SetFilter(onlyItemSets, true);
		banksets:SetFilter(hideEmpty, true);
		banksets:Hide();

	local bankgizmos = Container:New("Bank_Gizmos", { Bags = "bankframe+bank"; Name = L["Gizmos"]; Columns = 8; });
		bankgizmos:SetFilter(onlyBank, true);
		bankgizmos:SetFilter(onlyGizmos, true);
		bankgizmos:SetFilter(hideEmpty, true);
		bankgizmos:Hide();

	local bankarmor = Container:New("Bank_Armor", { Bags = "bankframe+bank"; Name = L["Armor"]; Columns = 8; });
		bankarmor:SetFilter(onlyBank, true);
		bankarmor:SetFilter(onlyArmor, true);
		bankarmor:SetFilter(hideEmpty, true);
		bankarmor:Hide();

	local bankweapons = Container:New("Bank_Weapons", { Bags = "bankframe+bank"; Name = L["Weapon"]; Columns = 8; });
		bankweapons:SetFilter(onlyBank, true);
		bankweapons:SetFilter(onlyWeapon, true);
		bankweapons:SetFilter(hideEmpty, true);
		bankweapons:Hide();

	local bankgems = Container:New("Bank_Gems", { Bags = "bankframe+bank"; Name = L["Gem"]; Columns = 8; });
		bankgems:SetFilter(onlyBank, true);
		bankgems:SetFilter(onlyGems, true);
		bankgems:SetFilter(hideEmpty, true);
		bankgems:Hide();

	local bankglyphs = Container:New("Bank_Glyphs", { Bags = "bankframe+bank"; Name = L["Glyph"]; Columns = 8; });
		bankglyphs:SetFilter(onlyBank, true);
		bankglyphs:SetFilter(onlyGlyphs, true);
		bankglyphs:SetFilter(hideEmpty, true);
		bankglyphs:Hide();

	local bankquest = Container:New("Bank_Quest", { Bags = "bankframe+bank"; Name = L["Quest"]; Columns = 8; });
		bankquest:SetFilter(onlyBank, true);
		bankquest:SetFilter(onlyQuest, true);
		bankquest:SetFilter(hideEmpty, true);
		bankquest:Hide();

	local bankconsumables = Container:New("Bank_Consumables", { Bags = "bankframe+bank"; Name = L["Consumables"]; Columns = 8; });
		bankconsumables:SetFilter(onlyBank, true);
		bankconsumables:SetFilter(onlyConsumables, true);
		bankconsumables:SetFilter(hideEmpty, true);
		bankconsumables:Hide();

	local bank = Container:New("Bank", { Bags = "bankframe+bank"; Name = L["Bank"]; Columns = 17; })
		bank:SetFilter(onlyBank, true);
		bank:Hide();
	

	local mainsets = Container:New("Main_Sets", { Bags = "backpack+bags"; Name = L["Sets"]; Columns = 9; });
		mainsets:SetFilter(onlyItemSets, true);
		mainsets:SetFilter(hideEmpty, true);

	local maingizmos = Container:New("Main_Gizmos", { Bags = "backpack+bags"; Name = L["Gizmos"]; Columns = 9; });
		maingizmos:SetFilter(onlyGizmos, true);
		maingizmos:SetFilter(hideEmpty, true);

	local keyring = Container:New("Keyring", { Bags = "backpack+bags"; Name = L["Keyring"]; Columns = 9; });
		keyring:SetFilter(onlyKeyring, true);
		keyring:SetFilter(hideEmpty, true);
		keyring:Hide(); -- we don't want to see the keyring by default, so hide it

	local mainjunk = Container:New("Main_Junk", { Bags = "backpack+bags"; Name = L["Junk"]; Columns = 9; });
		mainjunk:SetFilter(onlyJunk, true);
		mainjunk:SetFilter(hideEmpty, true);

	local mainarmor = Container:New("Main_Armor", { Bags = "backpack+bags"; Name = L["Armor"]; Columns = 9; });
		mainarmor:SetFilter(onlyArmor, true);
		mainarmor:SetFilter(hideJunk, true);
		mainarmor:SetFilter(hideEmpty, true);

	local mainweapons = Container:New("Main_Weapons", { Bags = "backpack+bags"; Name = L["Weapon"]; Columns = 9; });
		mainweapons:SetFilter(onlyWeapon, true);
		mainweapons:SetFilter(hideEmpty, true);

	local maingems = Container:New("Main_Gems", { Bags = "backpack+bags"; Name = L["Gem"]; Columns = 9; });
		maingems:SetFilter(onlyGems, true);
		maingems:SetFilter(hideEmpty, true);

	local mainglyphs = Container:New("Main_Glyphs", { Bags = "backpack+bags"; Name = L["Glyph"]; Columns = 9; });
		mainglyphs:SetFilter(onlyGlyphs, true);
		mainglyphs:SetFilter(hideEmpty, true);

	local mainquest = Container:New("Main_Quest", { Bags = "backpack+bags"; Name = L["Quest"]; Columns = 9; });
		mainquest:SetFilter(onlyQuest, true);
		mainquest:SetFilter(hideEmpty, true);

	local mainconsumables = Container:New("Main_Consumables", { Bags = "backpack+bags"; Name = L["Consumables"]; Columns = 9; });
		mainconsumables:SetFilter(onlyConsumables, true);
		mainconsumables:SetFilter(hideEmpty, true);

	local maintrade = Container:New("Main_Trade", { Bags = "backpack+bags"; Name = L["Trades"]; Columns = 9; });
		maintrade:SetFilter(onlyTradeGoods, true);
		maintrade:SetFilter(hideEmpty, true);

	local mainrecipe = Container:New("Main_Recipe", { Bags = "backpack+bags"; Name = L["Recipe"]; Columns = 9; });
		mainrecipe:SetFilter(onlyRecipe, true);
		mainrecipe:SetFilter(hideEmpty, true);

	local mainmisc = Container:New("Main_Misc", { Bags = "backpack+bags"; Name = L["Misc"]; Columns = 9; });
		mainmisc:SetFilter(onlyMisc, true);
		mainmisc:SetFilter(hideEmpty, true);
		mainmisc:SetFilter(hideSoulShards, true); -- soulshards default to the "misc" category, so hide them from it

	local ammo = Container:New("Main_Ammo", { Bags = "backpack+bags"; Name = L["Projectile"]; Columns = 9; });
		ammo:SetFilter(onlyAmmo, true);
		ammo:SetFilter(hideEmpty, true);

	local soulshards = Container:New("Main_SoulShards", { Bags = "backpack+bags"; Name = L["SoulShards"]; Columns = 9; });
		soulshards:SetFilter(onlySoulShards, true);
		soulshards:SetFilter(hideEmpty, true);

	local main = Container:New("Main", { Bags = "backpack+bags"; Name = L["Bags"]; Columns = 9; });
		main:SetFilter(onlyBags, true);
		main:SetFilter(hideAmmoBags, true);
		main:SetFilter(hideAmmo, true);

	-- position them
	-- intended to save their positions between sessions, but it keeps bugging out...?
	local dbag, dbank = defaults.bagspot, defaults.bankspot
	local sbag, sbank = gBags_DB.bagspot, gBags_DB.bankspot

	main:Place(dbag.point, UIParent, dbag.relpoint, dbag.x, dbag.y);
	bank:Place(dbank.point, UIParent, dbank.relpoint, dbank.x, dbank.y);
	
	bank:RegisterEvent("BANKFRAME_OPENED");
	bank:SetScript("OnEvent", function(self, event)
		Bags.atBank = true;
		gBags:OpenAll();
	end)
	
	-- position the subframes
	mainsets:Place(					"BOTTOMRIGHT", 	main, 				"BOTTOMLEFT", 	  0, 0);
	mainarmor:Place(				"BOTTOM", 		mainsets, 			"TOP", 			  0, 0);
	mainweapons:Place(				"BOTTOM", 		mainarmor, 			"TOP", 			  0, 0);
	maingems:Place(					"BOTTOM", 		mainweapons, 		"TOP", 			  0, 0);
	mainglyphs:Place(				"BOTTOM", 		maingems, 			"TOP", 			  0, 0);
	mainquest:Place(				"BOTTOM", 		mainglyphs, 		"TOP", 			  0, 0);
	maingizmos:Place(				"BOTTOM", 		mainquest, 			"TOP", 			  0, 0);
	mainjunk:Place(					"BOTTOM", 		maingizmos, 		"TOP", 			  0, 0);
	keyring:Place(					"BOTTOM", 		mainjunk, 			"TOP", 			  0, 0);
	mainconsumables:Place(			"BOTTOM", 		main, 				"TOP", 			  0, 0);
	maintrade:Place(				"BOTTOM", 		mainconsumables, 	"TOP", 			  0, 0);
	mainrecipe:Place(				"BOTTOM", 		maintrade, 			"TOP", 			  0, 0);
	mainmisc:Place(					"BOTTOM", 		mainrecipe, 		"TOP", 			  0, 0);
	ammo:Place(						"BOTTOM", 		mainmisc, 			"TOP", 			  0, 0);
	soulshards:Place(				"BOTTOM", 		ammo, 				"TOP", 			  0, 0);
	bankarmor:Place(				"TOPLEFT", 		bank, 				"BOTTOMLEFT", 	  0, 0);
	bankweapons:Place(				"TOP", 			bankarmor, 			"BOTTOM", 		  0, 0);
	bankgizmos:Place(				"TOPLEFT", 		bank, 				"TOPRIGHT", 	  0, 0);
	bankconsumables:Place(			"TOP", 			bankgizmos, 		"BOTTOM", 		  0, 0);
	banksets:Place(					"TOPRIGHT", 	bank, 				"BOTTOMRIGHT", 	  0, 0);
	bankgems:Place(					"TOP", 			banksets, 			"BOTTOM", 		  0, 0);
	bankglyphs:Place(				"TOP", 			bankgems, 			"BOTTOM", 	 	  0, 0);
	bankquest:Place(				"TOP", 			bankglyphs, 		"BOTTOM", 	 	  0, 0);
end
function Bags:OnOpen()
	gBags:OpenAllBags();
	if Bags:AtBank() then
		gBags:OpenAllBanks();
	end
end
function Bags:OnClose()
	gBags:CloseAll();
end

-- Rescale all the containers
function gBags:ScaleAll()
	Bags:GetContainer("Bank"):Scale();
	Bags:GetContainer("Bank_Armor"):Scale();
	Bags:GetContainer("Bank_Weapons"):Scale();
	Bags:GetContainer("Bank_Sets"):Scale();
	Bags:GetContainer("Bank_Consumables"):Scale();
	Bags:GetContainer("Bank_Gizmos"):Scale();
	Bags:GetContainer("Bank_Gems"):Scale();
	Bags:GetContainer("Bank_Glyphs"):Scale();
	Bags:GetContainer("Bank_Quest"):Scale();
	Bags:GetContainer("Main"):Scale();
	Bags:GetContainer("Main_Sets"):Scale();
	Bags:GetContainer("Main_Armor"):Scale();
	Bags:GetContainer("Main_Weapons"):Scale();
	Bags:GetContainer("Main_Gems"):Scale();
	Bags:GetContainer("Main_Glyphs"):Scale();
	Bags:GetContainer("Main_Quest"):Scale();
	Bags:GetContainer("Main_Gizmos"):Scale();
	Bags:GetContainer("Main_Junk"):Scale();
	Bags:GetContainer("Main_Consumables"):Scale();
	Bags:GetContainer("Main_Trade"):Scale();
	Bags:GetContainer("Main_Recipe"):Scale();
	Bags:GetContainer("Main_Misc"):Scale();
	Bags:GetContainer("Main_Ammo"):Scale();
	Bags:GetContainer("Main_SoulShards"):Scale();
	Bags:GetContainer("Keyring"):Scale();
end

-- opens backpack + bags
function gBags:OpenAllBags()
	Bags:GetContainer("Main_Sets"):Show(); 
	Bags:GetContainer("Main_Armor"):Show(); 
	Bags:GetContainer("Main_Weapons"):Show(); 
	Bags:GetContainer("Main_Gems"):Show(); 
	Bags:GetContainer("Main_Glyphs"):Show(); 
	Bags:GetContainer("Main_Quest"):Show(); 
	Bags:GetContainer("Main_Gizmos"):Show(); 
	if not(gBags_DB.hidejunk == 1) then
		Bags:GetContainer("Main_Junk"):Show(); 
	end
	Bags:GetContainer("Main_Consumables"):Show(); 
	Bags:GetContainer("Main_Trade"):Show(); 
	Bags:GetContainer("Main_Recipe"):Show(); 
	Bags:GetContainer("Main_Misc"):Show(); 
	if not((ammoCount() ~= 0) and (gBags_DB.hideammo == 1)) then 
		Bags:GetContainer("Main_Ammo"):Show(); 
	end
	if not(gBags_DB.hideshards == 1) then 
		Bags:GetContainer("Main_SoulShards"):Show();
	end
end

-- opens bankframe + bank bags
function gBags:OpenAllBanks()
	Bags:GetContainer("Bank_Quest"):Show(); 
	Bags:GetContainer("Bank_Glyphs"):Show(); 
	Bags:GetContainer("Bank_Gems"):Show(); 
	Bags:GetContainer("Bank_Consumables"):Show(); 
	Bags:GetContainer("Bank_Gizmos"):Show(); 
	Bags:GetContainer("Bank_Sets"):Show(); 
	Bags:GetContainer("Bank_Weapons"):Show(); 
	Bags:GetContainer("Bank_Armor"):Show(); 
	Bags:GetContainer("Bank"):Show();
end

-- opens everything
function gBags:OpenAll()
	self:OpenAllBags();
	self:OpenAllBanks();
end

-- closes everything
function gBags:CloseAll()
	Bags:GetContainer("Bank"):Hide();
	Bags:GetContainer("Bank_Armor"):Hide();
	Bags:GetContainer("Bank_Weapons"):Hide();
	Bags:GetContainer("Bank_Sets"):Hide();
	Bags:GetContainer("Bank_Consumables"):Hide();
	Bags:GetContainer("Bank_Gizmos"):Hide();
	Bags:GetContainer("Bank_Gems"):Hide();
	Bags:GetContainer("Bank_Glyphs"):Hide();
	Bags:GetContainer("Bank_Quest"):Hide();
	Bags:GetContainer("Main_Sets"):Hide();
	Bags:GetContainer("Main_Armor"):Hide();
	Bags:GetContainer("Main_Weapons"):Hide();
	Bags:GetContainer("Main_Gems"):Hide();
	Bags:GetContainer("Main_Glyphs"):Hide();
	Bags:GetContainer("Main_Quest"):Hide();
	Bags:GetContainer("Main_Gizmos"):Hide();
	Bags:GetContainer("Main_Junk"):Hide();
	Bags:GetContainer("Main_Consumables"):Hide();
	Bags:GetContainer("Main_Trade"):Hide();
	Bags:GetContainer("Main_Recipe"):Hide();
	Bags:GetContainer("Main_Misc"):Hide();
	Bags:GetContainer("Main_Ammo"):Hide();
	Bags:GetContainer("Main_SoulShards"):Hide();
end

-- addon events
function gBags:EQUIPMENT_SETS_CHANGED()
	cacheEQSets();
end

function gBags:OnInit()
	-- load settings
	gBags:ArgumentCheck( gBags_DB, defaults );
	db.scale = gBags_DB.scale
	
	-- hook our bags to the default open/close functions
	Bags:RegisterBlizzard();

end
function gBags:OnEnable()
	-- crazy little stunt to initialize bags upon login
	-- why? so that the bankframe won't bug out on us if we open that first
	cacheEQSets();
	OpenAllBags();
	CloseAllBags();
	
	-- Setup Interface Options panel
	self.optionspanel = self:CreateOptionsPanel({
		title = NAME;
		version = VERSION;
		panel = ((IsAddOnLoaded("gUI") and gUI.optionspanel) or nil);
		option = {
			[10] = {
				name = L["LOCK_FRAMES"];
				type = "button";
				value = (gBags_DB.locked == 1) and true or false;
				panelfunction = function() 
					gBags_DB.locked = (gBags_DB.locked == 1) and 0 or 1;
				end;
			};
			[20] = {
				name = L["HIDE_JUNK"];
				type = "button";
				value = (gBags_DB.hidejunk == 1) and true or false;
				panelfunction = function() 
					gBags_DB.hidejunk = (gBags_DB.hidejunk == 1) and 0 or 1;
					if (gBags_DB.hidejunk == 1) then
						if (Bags:GetContainer("Main_Junk"):IsShown()) then
							Bags:GetContainer("Main_Junk"):Hide();
						end
					elseif (gBags_DB.hidejunk == 0) then
						if (Bags:GetContainer("Main"):IsShown()) then
							Bags:GetContainer("Main_Junk"):Show();
						end
					end
				end;
			};
			[30] = {
				name = L["HIDE_SHARDS"];
				type = "button";
				value = (gBags_DB.hideshards == 1) and true or false;
				panelfunction = function() 
					gBags_DB.hideshards = (gBags_DB.hideshards == 1) and 0 or 1;
					if (gBags_DB.hideshards == 1) then
						if (Bags:GetContainer("Main_SoulShards"):IsShown()) then
							Bags:GetContainer("Main_SoulShards"):Hide();
						end
					elseif (gBags_DB.hideshards == 0) then
						if (Bags:GetContainer("Main"):IsShown()) then
							Bags:GetContainer("Main_SoulShards"):Show();
						end
					end
				end;
			};
			[40] = {
				name = L["HIDE_AMMO"];
				type = "button";
				value = (gBags_DB.hideammo == 1) and true or false;
				panelfunction = function() 
					gBags_DB.hideammo = (gBags_DB.hideammo == 1) and 0 or 1;
					if (gBags_DB.hideammo == 1) then
						if ((Bags:GetContainer("Main_Ammo"):IsShown()) and (ammoCount() ~= 0)) then
							Bags:GetContainer("Main_Ammo"):Hide();
						end
					elseif (gBags_DB.hideammo == 0) then
						if (Bags:GetContainer("Main"):IsShown()) then
							Bags:GetContainer("Main_Ammo"):Show();
						end
					end
				end;
			};
		};
	}, gBags_DB, self.defaults);
end
